-- -----------------------------------------------------
-- OpenEstate-SQL routines for HSQLDB
-- Copyright (C) 2009-2018 OpenEstate.org
-- -----------------------------------------------------
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_contacts
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_contacts( val_contact_id BIGINT )
  SPECIFIC remove_openestate_contacts
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Ansprechpartner entfernen
    DELETE
      FROM openestate_contacts
      WHERE contact_id = val_contact_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_geodbs
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_geodbs( val_geodb_id BIGINT )
  SPECIFIC remove_openestate_geodbs
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Geo-DB entfernen
    DELETE
      FROM openestate_geodbs
      WHERE geodb_id = val_geodb_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_geodbs_areas
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_geodbs_areas( val_area_id BIGINT )
  SPECIFIC remove_openestate_geodbs_areas
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Region entfernen
    DELETE
      FROM openestate_geodbs_areas
      WHERE area_id = val_area_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_geodbs_countries
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_geodbs_countries( val_country_id BIGINT )
  SPECIFIC remove_openestate_geodbs_countries
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Land entfernen
    DELETE
      FROM openestate_geodbs_countries
      WHERE country_id = val_country_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_languages
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_languages( val_language_id BIGINT )
  SPECIFIC remove_openestate_languages
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Sprache entfernen
    DELETE
      FROM openestate_languages
      WHERE language_id = val_language_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects( val_object_id BIGINT )
  SPECIFIC remove_openestate_objects
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Objekt entfernen
    DELETE
      FROM openestate_objects
      WHERE object_id = val_object_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_attribs
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_attribs(
    val_object_id BIGINT,
    val_language_id BIGINT,
    val_attrib_name VARCHAR(255)
  )
  SPECIFIC remove_openestate_objects_attribs
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Attribut in einer Sprache entfernen
    IF val_language_id IS NOT NULL AND val_attrib_name IS NOT NULL THEN

      DELETE
        FROM openestate_objects_attribs
        WHERE object_id = val_object_id
        AND language_id = val_language_id
        AND attrib_name = val_attrib_name;

    -- Attribut in allen Sprachen entfernen
    ELSEIF val_attrib_name IS NOT NULL THEN

      DELETE
        FROM openestate_objects_attribs
        WHERE object_id = val_object_id
        AND attrib_name = val_attrib_name;

    -- Attribute in allen Sprache entfernen
    ELSEIF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_attribs
        WHERE object_id = val_object_id
        AND language_id = val_language_id;

    -- Alle Attribute entfernen
    ELSE

      DELETE
        FROM openestate_objects_attribs
        WHERE object_id = val_object_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_files
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_files( val_file_id BIGINT )
  SPECIFIC remove_openestate_objects_files
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Datei entfernen
    DELETE
      FROM openestate_objects_files
      WHERE file_id = val_file_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_files_i18n
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_files_i18n(
    val_file_id BIGINT,
    val_language_id BIGINT
  )
  SPECIFIC remove_openestate_objects_files_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Übersetzungen zu einer Sprache entfernen
    IF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_files_i18n
        WHERE file_id = val_file_id
        AND language_id = val_language_id;

    -- Alle Übersetzungen entfernen
    ELSE

      DELETE
        FROM openestate_objects_files_i18n
        WHERE file_id = val_file_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_geodbs_areas
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_geodbs_areas(
    val_object_id BIGINT,
    val_area_id BIGINT
  )
  SPECIFIC remove_openestate_objects_geodbs_areas
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_area_id IS NOT NULL AND val_area_id > 0 THEN
      DELETE
        FROM openestate_objects_geodbs_areas
        WHERE object_id = val_object_id
        AND area_id = val_area_id;

    ELSE
      DELETE
        FROM openestate_objects_geodbs_areas
        WHERE object_id = val_object_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_i18n
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_i18n(
    val_object_id BIGINT,
    val_language_id BIGINT
  )
  SPECIFIC remove_openestate_objects_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Übersetzungen zu einer Sprache entfernen
    IF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_i18n
        WHERE object_id = val_object_id
        AND language_id = val_language_id;

    -- Alle Übersetzungen entfernen
    ELSE

      DELETE
        FROM openestate_objects_i18n
        WHERE object_id = val_object_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_links
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_links( val_link_id BIGINT )
  SPECIFIC remove_openestate_objects_links
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Datei entfernen
    DELETE
      FROM openestate_objects_links
      WHERE link_id = val_link_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_links_i18n
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_links_i18n(
    val_link_id BIGINT,
    val_language_id BIGINT
  )
  SPECIFIC remove_openestate_objects_links_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Übersetzungen zu einer Sprache entfernen
    IF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_links_i18n
        WHERE link_id = val_link_id
        AND language_id = val_language_id;

    -- Alle Übersetzungen entfernen
    ELSE

      DELETE
        FROM openestate_objects_links_i18n
        WHERE link_id = val_link_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_searches
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_searches( val_search_id BIGINT )
  SPECIFIC remove_openestate_searches
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Suche entfernen
    DELETE
      FROM openestate_searches
      WHERE search_id = val_search_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_searches_attribs
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_searches_attribs( val_search_id BIGINT )
  SPECIFIC remove_openestate_searches_attribs
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Kriterien entfernen
    DELETE
      FROM openestate_searches_attribs
      WHERE search_id = val_search_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_searches_params
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_searches_params(
    val_search_id BIGINT,
    val_param_key VARCHAR(255)
  )
  SPECIFIC remove_openestate_searches_params
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Spezifischen Parameter entfernen
    IF val_param_key IS NOT NULL THEN

      DELETE
        FROM openestate_searches_params
        WHERE search_id = val_search_id
        AND param_key = val_param_key;

    -- Alle Parameter entfernen
    ELSE

      DELETE
        FROM openestate_searches_params
        WHERE search_id = val_search_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_openestate_searches_values
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_searches_values( val_search_id BIGINT )
  SPECIFIC remove_openestate_searches_values
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Kriterien entfernen
    DELETE
      FROM openestate_searches_values
      WHERE search_id = val_search_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Function get_openestate_objects_id
-- -----------------------------------------------------
CREATE FUNCTION get_openestate_objects_id( val_public_id VARCHAR(255) )
  RETURNS BIGINT
  SPECIFIC get_openestate_objects_id
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE object_id BIGINT;

    -- Objekt-ID aus Objekt-Nr ermitteln
    SELECT object_id
      INTO object_id
      FROM openestate_objects
      WHERE object_public_id IS NOT NULL AND object_public_id = val_public_id;

    -- ID der ermittelten Immobilie zurückliefern
    RETURN object_id;
  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_contacts
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_contacts(
    INOUT val_contact_id BIGINT,
    val_person_first_name VARCHAR(150),
    val_person_middle_name VARCHAR(150),
    val_person_last_name VARCHAR(150),
    val_person_title VARCHAR(50),
    val_person_gender CHAR(1),
    val_person_date_of_birth DATE,
    val_person_mail VARCHAR(150),
    val_person_phone VARCHAR(50),
    val_person_mobile_phone VARCHAR(50),
    val_person_fax VARCHAR(50),
    val_address_street VARCHAR(150),
    val_address_street2 VARCHAR(150),
    val_address_street_nr VARCHAR(20),
    val_address_postal_code VARCHAR(10),
    val_address_city VARCHAR(150),
    val_address_district VARCHAR(150),
    val_address_country CHAR(2),
    val_address_region VARCHAR(150),
    val_address_latitude DECIMAL(15,12),
    val_address_longitude DECIMAL(15,12),
    val_contact_website VARCHAR(255),
    val_contact_office_hours LONGVARCHAR
  )
  SPECIFIC save_openestate_contacts
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Neuer Ansprechpartner
    IF val_contact_id < 1 THEN

      -- Ansprechpartner speichern
      INSERT INTO openestate_contacts (
        person_first_name,
        person_middle_name,
        person_last_name,
        person_title,
        person_gender,
        person_date_of_birth,
        person_mail,
        person_phone,
        person_mobile_phone,
        person_fax,
        address_street,
        address_street2,
        address_street_nr,
        address_postal_code,
        address_city,
        address_district,
        address_country,
        address_region,
        address_latitude,
        address_longitude,
        contact_website,
        contact_office_hours )
      VALUES (
        val_person_first_name,
        val_person_middle_name,
        val_person_last_name,
        val_person_title,
        val_person_gender,
        val_person_date_of_birth,
        val_person_mail,
        val_person_phone,
        val_person_mobile_phone,
        val_person_fax,
        val_address_street,
        val_address_street2,
        val_address_street_nr,
        val_address_postal_code,
        val_address_city,
        val_address_district,
        val_address_country,
        val_address_region,
        val_address_latitude,
        val_address_longitude,
        val_contact_website,
        val_contact_office_hours );

      -- ID zurückliefern
      SET val_contact_id = IDENTITY();

    ELSE

      -- Ansprechpartner bearbeiten
      UPDATE openestate_contacts
        SET
          person_first_name = val_person_first_name,
          person_middle_name = val_person_middle_name,
          person_last_name = val_person_last_name,
          person_title = val_person_title,
          person_gender = val_person_gender,
          person_date_of_birth = val_person_date_of_birth,
          person_mail = val_person_mail,
          person_phone = val_person_phone,
          person_mobile_phone = val_person_mobile_phone,
          person_fax = val_person_fax,
          address_street = val_address_street,
          address_street2 = val_address_street2,
          address_street_nr = val_address_street_nr,
          address_postal_code = val_address_postal_code,
          address_city = val_address_city,
          address_district = val_address_district,
          address_country = val_address_country,
          address_region = val_address_region,
          address_latitude = val_address_latitude,
          address_longitude = val_address_longitude,
          contact_website = val_contact_website,
          contact_office_hours = val_contact_office_hours,
          modified_at = NOW()
        WHERE
          contact_id = val_contact_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_geodbs
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_geodbs(
    INOUT val_geodb_id BIGINT,
    val_geodb_name VARCHAR(255),
    val_geodb_provider VARCHAR(255),
    val_geodb_build_time TIMESTAMP WITH TIME ZONE
  )
  SPECIFIC save_openestate_geodbs
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- neue Geo-DB erzeugen
    IF val_geodb_id < 1 THEN
      INSERT INTO openestate_geodbs (
        geodb_name,
        geodb_provider,
        geodb_build_time )
      VALUES (
        val_geodb_name,
        val_geodb_provider,
        val_geodb_build_time );

      -- ID zurückliefern
      SET val_geodb_id = IDENTITY();

    -- bestehende Geo-DB bearbeiten
    ELSE
      UPDATE openestate_geodbs
        SET
          geodb_name = val_geodb_name,
          geodb_provider = val_geodb_provider,
          geodb_build_time = val_geodb_build_time,
          modified_at = NOW()
        WHERE
          geodb_id = val_geodb_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_geodbs_areas
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_geodbs_areas(
    INOUT val_area_id BIGINT,
    val_country_id BIGINT,
    val_area_parent_id BIGINT,
    val_area_name VARCHAR(255),
    val_area_level VARCHAR(15),
    val_area_provider_id VARCHAR(255)
  )
  SPECIFIC save_openestate_geodbs_areas
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- neue Region erzeugen
    IF val_area_id < 1 THEN
      INSERT INTO openestate_geodbs_areas (
        country_id,
        area_parent_id,
        area_name,
        area_level,
        area_provider_id )
      VALUES (
        val_country_id,
        val_area_parent_id,
        val_area_name,
        val_area_level,
        val_area_provider_id );

      -- ID zurückliefern
      SET val_area_id = IDENTITY();

    -- bestehende Region bearbeiten
    ELSE
      UPDATE openestate_geodbs_areas
        SET
          country_id = val_country_id,
          area_parent_id = val_area_parent_id,
          area_name = val_area_name,
          area_level = val_area_level,
          area_provider_id = val_area_provider_id
        WHERE
          area_id = val_area_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_geodbs_countries
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_geodbs_countries(
    INOUT val_country_id BIGINT,
    val_geodb_id BIGINT,
    val_country_iso CHAR(2),
    val_country_name VARCHAR(255),
    val_country_provider_id VARCHAR(255)
  )
  SPECIFIC save_openestate_geodbs_countries
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- neues Land erzeugen
    IF val_country_id < 1 THEN
      INSERT INTO openestate_geodbs_countries (
        geodb_id,
        country_iso,
        country_name,
        country_provider_id )
      VALUES (
        val_geodb_id,
        val_country_iso,
        val_country_name,
        val_country_provider_id );

      -- ID zurückliefern
      SET val_country_id = IDENTITY();

    -- bestehendes Land bearbeiten
    ELSE
      UPDATE openestate_geodbs_countries
        SET
          geodb_id = val_geodb_id,
          country_iso = val_country_iso,
          country_name = val_country_name,
          country_provider_id = val_country_provider_id
        WHERE
          country_id = val_country_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_languages
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_languages(
    INOUT val_language_id BIGINT,
    val_language_code VARCHAR(50),
    val_language_enabled BOOLEAN
  )
  SPECIFIC save_openestate_languages
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Neue Sprache
    IF val_language_id < 1 THEN

      -- Sprache speichern
      INSERT INTO openestate_languages (
        language_code,
        language_enabled )
      VALUES (
        val_language_code,
        val_language_enabled );

      -- ID zurückliefern
      SET val_language_id = IDENTITY();

    ELSE

      -- Sprache bearbeiten
      UPDATE openestate_languages
        SET
          language_code = val_language_code,
          language_enabled = val_language_enabled,
          modified_at = NOW()
        WHERE
          language_id = val_language_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects(
    INOUT val_object_id BIGINT,
    val_contact_id BIGINT,
    val_object_type VARCHAR(100),
    val_object_action VARCHAR(100),
    val_object_public_id VARCHAR(255),
    val_object_group_id VARCHAR(255),
    val_object_currency CHAR(3),
    val_object_hidden_prices BOOLEAN,
    val_object_publish_begin TIMESTAMP,
    val_object_publish_end TIMESTAMP,
    val_address_published BOOLEAN,
    val_address_street VARCHAR(150),
    val_address_street2 VARCHAR(150),
    val_address_street_nr VARCHAR(20),
    val_address_postal_code VARCHAR(10),
    val_address_city VARCHAR(150),
    val_address_district VARCHAR(150),
    val_address_country CHAR(2),
    val_address_region VARCHAR(150),
    val_address_latitude DECIMAL(15,12),
    val_address_longitude DECIMAL(15,12)
  )
  SPECIFIC save_openestate_objects
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Neues Objekt
    IF val_object_id < 1 THEN

      -- Objekt speichern
      INSERT INTO openestate_objects (
        contact_id,
        object_type,
        object_action,
        object_public_id,
        object_group_id,
        object_currency,
        object_hidden_prices,
        object_publish_begin,
        object_publish_end,
        address_published,
        address_street,
        address_street2,
        address_street_nr,
        address_postal_code,
        address_city,
        address_district,
        address_country,
        address_region,
        address_latitude,
        address_longitude )
      VALUES (
        val_contact_id,
        val_object_type,
        val_object_action,
        val_object_public_id,
        val_object_group_id,
        val_object_currency,
        val_object_hidden_prices,
        val_object_publish_begin,
        val_object_publish_end,
        val_address_published,
        val_address_street,
        val_address_street2,
        val_address_street_nr,
        val_address_postal_code,
        val_address_city,
        val_address_district,
        val_address_country,
        val_address_region,
        val_address_latitude,
        val_address_longitude );

      -- ID zurückliefern
      SET val_object_id = IDENTITY();

    ELSE

      -- Objekt bearbeiten
      UPDATE openestate_objects
        SET
          contact_id = val_contact_id,
          object_type = val_object_type,
          object_action = val_object_action,
          object_public_id = val_object_public_id,
          object_group_id = val_object_group_id,
          object_currency = val_object_currency,
          object_hidden_prices = val_object_hidden_prices,
          object_publish_begin = val_object_publish_begin,
          object_publish_end = val_object_publish_end,
          address_published = val_address_published,
          address_street = val_address_street,
          address_street2 = val_address_street2,
          address_street_nr = val_address_street_nr,
          address_postal_code = val_address_postal_code,
          address_city = val_address_city,
          address_district = val_address_district,
          address_country = val_address_country,
          address_region = val_address_region,
          address_latitude = val_address_latitude,
          address_longitude = val_address_longitude,
          modified_at = NOW()
        WHERE
          object_id = val_object_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_attribs
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_attribs(
    val_object_id BIGINT,
    val_language_id BIGINT,
    val_attrib_name VARCHAR(255),
    val_attrib_unit VARCHAR(25),
    val_value_text LONGVARCHAR,
    val_value_string VARCHAR(255),
    val_value_boolean BOOLEAN,
    val_value_numeric DECIMAL(15,2),
    val_value_numeric_base DECIMAL(15,2),
    val_value_date DATE,
    val_value_time TIME,
    val_value_datetime TIMESTAMP WITH TIME ZONE
  )
  SPECIFIC save_openestate_objects_attribs
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Attribut speichern
    INSERT INTO openestate_objects_attribs (
      object_id,
      language_id,
      attrib_name,
      attrib_unit,
      value_text,
      value_string,
      value_boolean,
      value_numeric,
      value_numeric_base,
      value_date,
      value_time,
      value_datetime )
    VALUES (
      val_object_id,
      val_language_id,
      val_attrib_name,
      val_attrib_unit,
      val_value_text,
      val_value_string,
      val_value_boolean,
      val_value_numeric,
      val_value_numeric_base,
      val_value_date,
      val_value_time,
      val_value_datetime );

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_files
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_files(
    INOUT val_file_id BIGINT,
    val_object_id BIGINT,
    val_file_name VARCHAR(255),
    val_file_url VARCHAR(255),
    val_file_content BLOB(1G),
    val_file_content_preview BLOB(1M),
    val_file_mimetype VARCHAR(255),
    val_file_published BOOLEAN,
    val_file_type VARCHAR(100),
    val_file_size BIGINT,
    val_file_hash VARCHAR(40),
    val_file_image_width INT,
    val_file_image_height INT,
    val_file_position INT
  )
  SPECIFIC save_openestate_objects_files
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neue Datei
    IF val_file_id < 1 THEN

      -- Datei speichern
      INSERT INTO openestate_objects_files (
        object_id,
        file_name,
        file_url,
        file_mimetype,
        file_published,
        file_type,
        file_size,
        file_hash,
        file_image_width,
        file_image_height,
        file_position )
      VALUES (
        val_object_id,
        val_file_name,
        val_file_url,
        val_file_mimetype,
        val_file_published,
        val_file_type,
        val_file_size,
        val_file_hash,
        val_file_image_width,
        val_file_image_height,
        val_file_position );

      -- ID zurückliefern
      SET val_file_id = IDENTITY();

    ELSE

      -- Datei bearbeiten
      UPDATE openestate_objects_files
        SET
          object_id = val_object_id,
          file_name = val_file_name,
          file_url = val_file_url,
          file_mimetype = val_file_mimetype,
          file_published = val_file_published,
          file_type = val_file_type,
          file_size = val_file_size,
          file_hash = val_file_hash,
          file_image_width = val_file_image_width,
          file_image_height = val_file_image_height,
          file_position = val_file_position,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

    -- Datei ggf. aktualisieren
    IF val_file_content IS NOT NULL THEN

      -- Datei bearbeiten
      UPDATE openestate_objects_files
        SET
          file_content = val_file_content,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

    -- Vorschau ggf. aktualisieren
    IF val_file_content_preview IS NOT NULL THEN

      -- Vorschau bearbeiten
      UPDATE openestate_objects_files
        SET
          file_content_preview = val_file_content_preview,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_files_i18n
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_files_i18n(
    val_file_id BIGINT,
    val_language_id BIGINT,
    val_i18n_title VARCHAR(255)
  )
  SPECIFIC save_openestate_objects_files_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzung ggf. entfernen
    DELETE
      FROM openestate_objects_files_i18n
      WHERE file_id = val_file_id
      AND language_id = val_language_id;

    -- Übersetzung speichern
    INSERT INTO openestate_objects_files_i18n (
      file_id,
      language_id,
      i18n_title )
    VALUES (
      val_file_id,
      val_language_id,
      val_i18n_title );

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_geodbs_areas
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_geodbs_areas(
    val_object_id BIGINT,
    val_area_id BIGINT
  )
  SPECIFIC save_openestate_objects_geodbs_areas
  MODIFIES SQL DATA
  BEGIN ATOMIC

    DELETE
      FROM openestate_objects_geodbs_areas
      WHERE object_id = val_object_id
      AND area_id = val_area_id;

    INSERT
      INTO openestate_objects_geodbs_areas (
        object_id,
        area_id)
      VALUES (
        val_object_id,
        val_area_id);

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_i18n
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_i18n(
    val_object_id BIGINT,
    val_language_id BIGINT,
    val_i18n_title VARCHAR(255)
  )
  SPECIFIC save_openestate_objects_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzung ggf. entfernen
    DELETE
      FROM openestate_objects_i18n
      WHERE object_id = val_object_id
      AND language_id = val_language_id;

    -- Übersetzung speichern
    INSERT INTO openestate_objects_i18n (
      object_id,
      language_id,
      i18n_title )
    VALUES (
      val_object_id,
      val_language_id,
      val_i18n_title );

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_links
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_links(
    INOUT val_link_id BIGINT,
    val_object_id BIGINT,
    val_link_web_url VARCHAR(255),
    val_link_provider_id VARCHAR(255),
    val_link_provider_name VARCHAR(255),
    val_link_published BOOLEAN,
    val_link_position INT
  )
  SPECIFIC save_openestate_objects_links
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuer Link
    IF val_link_id < 1 THEN

      -- Datei speichern
      INSERT INTO openestate_objects_links (
        object_id,
        link_web_url,
        link_provider_id,
        link_provider_name,
        link_published,
        link_position )
      VALUES (
        val_object_id,
        val_link_web_url,
        val_link_provider_id,
        val_link_provider_name,
        val_link_published,
        val_link_position );

      -- ID zurückliefern
      SET val_link_id = IDENTITY();

    ELSE

      -- Link bearbeiten
      UPDATE openestate_objects_links
        SET
          object_id = val_object_id,
          link_web_url = val_link_web_url,
          link_provider_id = val_link_provider_id,
          link_provider_name = val_link_provider_name,
          link_published = val_link_published,
          link_position = val_link_position,
          modified_at = NOW()
        WHERE
          link_id = val_link_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_links_i18n
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_links_i18n(
    val_link_id BIGINT,
    val_language_id BIGINT,
    val_i18n_title VARCHAR(255)
  )
  SPECIFIC save_openestate_objects_links_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_link_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzung ggf. entfernen
    DELETE
      FROM openestate_objects_links_i18n
      WHERE link_id = val_link_id
      AND language_id = val_language_id;

    -- Übersetzung speichern
    INSERT INTO openestate_objects_links_i18n (
      link_id,
      language_id,
      i18n_title )
    VALUES (
      val_link_id,
      val_language_id,
      val_i18n_title );

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_searches
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_searches(
    INOUT val_search_id BIGINT,
    val_search_name VARCHAR(255),
    val_search_notes LONGVARCHAR,
    val_search_enabled BOOLEAN
  )
  SPECIFIC save_openestate_searches
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Neue Suche
    IF val_search_id < 1 THEN

      -- Suche speichern
      INSERT INTO openestate_searches (
        search_name,
        search_notes,
        search_enabled )
      VALUES (
        val_search_name,
        val_search_notes,
        val_search_enabled );

      -- ID zurückliefern
      SET val_search_id = IDENTITY();

    ELSE

      -- Suche bearbeiten
      UPDATE openestate_searches
        SET
          search_name = val_search_name,
          search_notes = val_search_notes,
          search_enabled = val_search_enabled,
          modified_at = NOW()
        WHERE
          search_id = val_search_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_searches_attribs
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_searches_attribs(
    val_search_id BIGINT,
    val_attrib_name VARCHAR(255),
    val_attrib_equals VARCHAR(255),
    val_attrib_equals_like BOOLEAN,
    val_attrib_equals_case BOOLEAN,
    val_attrib_matches VARCHAR(255),
    val_attrib_numeric_min DECIMAL(15,2),
    val_attrib_numeric_max DECIMAL(15,2),
    val_attrib_numeric_unit VARCHAR(20),
    val_attrib_time_begin TIMESTAMP WITH TIME ZONE,
    val_attrib_time_end TIMESTAMP WITH TIME ZONE,
    val_attrib_order BIGINT
  )
  SPECIFIC save_openestate_searches_attribs
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Kriterium speichern
    INSERT INTO openestate_searches_attribs (
      search_id,
      attrib_name,
      attrib_equals,
      attrib_equals_like,
      attrib_equals_case,
      attrib_matches,
      attrib_numeric_min,
      attrib_numeric_max,
      attrib_numeric_unit,
      attrib_time_begin,
      attrib_time_end,
      attrib_order )
    VALUES (
      val_search_id,
      val_attrib_name,
      val_attrib_equals,
      val_attrib_equals_like,
      val_attrib_equals_case,
      val_attrib_matches,
      val_attrib_numeric_min,
      val_attrib_numeric_max,
      val_attrib_numeric_unit,
      val_attrib_time_begin,
      val_attrib_time_end,
      val_attrib_order );

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_searches_params
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_searches_params(
    val_search_id BIGINT,
    val_param_key VARCHAR(255),
    val_param_value VARCHAR(255)
  )
  SPECIFIC save_openestate_searches_params
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- existierenden Paramter ggf. entfernen
    IF val_param_key IS NOT NULL THEN

      DELETE
        FROM openestate_searches_params
        WHERE search_id = val_search_id
        AND param_key = val_param_key;

    END IF;

    -- Parameter speichern
    INSERT INTO openestate_searches_params (
      search_id,
      param_key,
      param_value )
    VALUES (
      val_search_id,
      val_param_key,
      val_param_value );

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure save_openestate_searches_values
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_searches_values(
    val_search_id BIGINT,
    val_value_type VARCHAR(100),
    val_value_equals VARCHAR(255),
    val_value_equals_like BOOLEAN,
    val_value_equals_case BOOLEAN,
    val_value_order BIGINT
  )
  SPECIFIC save_openestate_searches_values
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Kriterium speichern
    INSERT INTO openestate_searches_values (
      search_id,
      value_type,
      value_equals,
      value_equals_like,
      value_equals_case,
      value_order )
    VALUES (
      val_search_id,
      val_value_type,
      val_value_equals,
      val_value_equals_like,
      val_value_equals_case,
      val_value_order );

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure set_openestate_languages_enabled
-- -----------------------------------------------------
CREATE PROCEDURE set_openestate_languages_enabled(
    val_language_id BIGINT,
    val_language_enabled BOOLEAN
  )
  SPECIFIC set_openestate_languages_enabled
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_language_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Sprache bearbeiten
    UPDATE openestate_languages
      SET
        language_enabled = val_language_enabled,
        modified_at = NOW()
      WHERE
        language_id = val_language_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure set_openestate_objects_contact
-- -----------------------------------------------------
CREATE PROCEDURE set_openestate_objects_contact(
    val_object_id BIGINT,
    val_contact_id BIGINT
  )
  SPECIFIC set_openestate_objects_contact
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Objekt bearbeiten
    UPDATE openestate_objects
      SET
        contact_id = val_contact_id,
        modified_at = NOW()
      WHERE
        object_id = val_object_id;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure set_openestate_objects_files_content
-- -----------------------------------------------------
CREATE PROCEDURE set_openestate_objects_files_content(
    val_file_id BIGINT,
    val_file_size BIGINT,
    val_file_hash VARCHAR(40),
    val_file_content BLOB(1G),
    val_file_content_preview BLOB(1M)
  )
  SPECIFIC set_openestate_objects_files_content
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Datei ggf. aktualisieren
    IF val_file_content IS NOT NULL THEN

      -- Datei bearbeiten
      UPDATE openestate_objects_files
        SET
          file_content = val_file_content,
          file_size = val_file_size,
          file_hash = val_file_hash,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

    -- Vorschau ggf. aktualisieren
    IF val_file_content_preview IS NOT NULL THEN

      -- Vorschau bearbeiten
      UPDATE openestate_objects_files
        SET
          file_content_preview = val_file_content_preview,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

  END;
.;
\.
-- -----------------------------------------------------
-- Procedure set_openestate_searches_enabled
-- -----------------------------------------------------
CREATE PROCEDURE set_openestate_searches_enabled(
    val_search_id BIGINT,
    val_search_enabled BOOLEAN
  )
  SPECIFIC set_openestate_searches_enabled
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Suche bearbeiten
    UPDATE openestate_searches
      SET
        search_enabled = val_search_enabled,
        modified_at = NOW()
      WHERE
        search_id = val_search_id;

  END;
.;
