-- -----------------------------------------------------
-- Mail-Addon for OpenEstate-ImmoTool
-- schema for HSQLDB
-- Copyright (C) 2009-2018 OpenEstate.org
-- -----------------------------------------------------

CREATE SEQUENCE seq_immotool_mails_accounts
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_immotool_mails_folders
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_immotool_mails_messages
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_immotool_mails_parts
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;


-- -----------------------------------------------------
-- Table immotool_mails_accounts
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_mails_accounts (
  account_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_mails_accounts NOT NULL,
  account_name VARCHAR(100) NOT NULL,
  account_type VARCHAR(50) NOT NULL,
  account_notes LONGVARCHAR DEFAULT NULL NULL,
  account_enabled BOOLEAN DEFAULT true NOT NULL,
  account_auto_retrieval BOOLEAN DEFAULT false NOT NULL,
  account_auto_retrieval_interval BIGINT DEFAULT 0 NOT NULL,
  account_auto_retrieval_at TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  access_owner_id BIGINT DEFAULT 1 NOT NULL,
  access_group_id BIGINT DEFAULT 1 NOT NULL,
  access_permissions INT DEFAULT 0 NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (account_id),
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT
);


-- -----------------------------------------------------
-- Table immotool_mails_accounts_settings
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_mails_accounts_settings (
  account_id BIGINT NOT NULL,
  setting_key VARCHAR(255) NOT NULL,
  setting_value LONGVARCHAR DEFAULT NULL NULL,
  PRIMARY KEY (account_id, setting_key),
  FOREIGN KEY (account_id)
    REFERENCES immotool_mails_accounts(account_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_mails_folders
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_mails_folders (
  folder_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_mails_folders NOT NULL,
  account_id BIGINT NOT NULL,
  folder_parent_id BIGINT DEFAULT NULL NULL,
  folder_name VARCHAR(255) DEFAULT NULL NULL,
  folder_holds_messages BOOLEAN DEFAULT FALSE NOT NULL,
  folder_holds_folders BOOLEAN DEFAULT FALSE NOT NULL,
  folder_auto_retrieval BOOLEAN DEFAULT FALSE NOT NULL,
  folder_special_type VARCHAR(50) DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (folder_id),
  UNIQUE (account_id, folder_parent_id, folder_name),
  FOREIGN KEY (folder_parent_id)
    REFERENCES immotool_mails_folders(folder_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (account_id)
    REFERENCES immotool_mails_accounts(account_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_mails_messages
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_mails_messages (
  message_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_mails_messages NOT NULL,
  folder_id BIGINT NOT NULL,
  message_subject VARCHAR(255) NOT NULL,
  message_from VARCHAR(255) DEFAULT NULL NULL,
  message_to VARCHAR(255) DEFAULT NULL NULL,
  message_size BIGINT DEFAULT 0 NOT NULL,
  message_uid BIGINT DEFAULT NULL NULL,
  message_uid_validity BIGINT DEFAULT NULL NULL,
  message_mime_id VARCHAR(255) DEFAULT NULL NULL,
  message_hash VARCHAR(255) DEFAULT NULL NULL,
  message_in_reply_to VARCHAR(255) DEFAULT NULL NULL,
  message_is_answered BOOLEAN DEFAULT false NOT NULL,
  message_is_archived BOOLEAN DEFAULT false NOT NULL,
  message_is_downloaded BOOLEAN DEFAULT false NOT NULL,
  message_is_draft BOOLEAN DEFAULT false NOT NULL,
  message_is_flagged BOOLEAN DEFAULT false NOT NULL,
  message_is_forwarded BOOLEAN DEFAULT false NOT NULL,
  message_is_seen BOOLEAN DEFAULT false NOT NULL,
  message_is_received BOOLEAN DEFAULT false NOT NULL,
  message_has_attachments BOOLEAN DEFAULT false NOT NULL,
  message_mdn_status VARCHAR(25) DEFAULT NULL NULL,
  message_sent_at TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  message_received_at TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (message_id),
  FOREIGN KEY (folder_id)
    REFERENCES immotool_mails_folders(folder_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_mails_parts
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_mails_parts (
  part_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_mails_parts NOT NULL,
  message_id BIGINT NOT NULL,
  part_parent_id BIGINT DEFAULT NULL NULL,
  part_type VARCHAR(255) NOT NULL,
  part_description VARCHAR(255) DEFAULT NULL NULL,
  part_disposition VARCHAR(255) DEFAULT NULL NULL,
  part_text LONGVARCHAR DEFAULT NULL NULL,
  part_file_name VARCHAR(255) DEFAULT NULL NULL,
  part_file_size BIGINT DEFAULT NULL NULL,
  part_file_content BLOB(1G) DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (part_id),
  FOREIGN KEY (part_parent_id)
    REFERENCES immotool_mails_parts(part_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (message_id)
    REFERENCES immotool_mails_messages(message_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_mails_headers
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_mails_headers (
  part_id BIGINT NOT NULL,
  header_key VARCHAR(255) NOT NULL,
  header_value LONGVARCHAR NOT NULL,
  FOREIGN KEY (part_id)
    REFERENCES immotool_mails_parts(part_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_mails_addresses
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_mails_addresses (
  message_id BIGINT NOT NULL,
  address_type VARCHAR(8) NOT NULL,
  address_value VARCHAR(255) NOT NULL,
  address_person VARCHAR(255) DEFAULT NULL NULL,
  FOREIGN KEY (message_id)
    REFERENCES immotool_mails_messages(message_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- View view_immotool_mails_accounts
-- -----------------------------------------------------
CREATE VIEW view_immotool_mails_accounts AS
  SELECT * FROM immotool_mails_accounts
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(access_permissions, 64) = 64)
    OR
    (BITAND(access_permissions, 8) = 8 AND access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(access_permissions, 1) = 1 AND access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_immotool_mails_accounts
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_mails_accounts_settings
-- -----------------------------------------------------
CREATE VIEW view_immotool_mails_accounts_settings AS
  SELECT * FROM immotool_mails_accounts_settings
  WHERE account_id IN (SELECT account_id FROM view_immotool_mails_accounts);

GRANT SELECT
  ON view_immotool_mails_accounts_settings
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_mails_folders
-- -----------------------------------------------------
CREATE VIEW view_immotool_mails_folders AS
  SELECT * FROM immotool_mails_folders
  WHERE account_id IN (SELECT account_id FROM view_immotool_mails_accounts);

GRANT SELECT
  ON view_immotool_mails_folders
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_mails_messages
-- -----------------------------------------------------
CREATE VIEW view_immotool_mails_messages AS
  SELECT * FROM immotool_mails_messages
  WHERE folder_id IN (SELECT folder_id FROM view_immotool_mails_folders);

GRANT SELECT
  ON view_immotool_mails_messages
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_mails_parts
-- -----------------------------------------------------
CREATE VIEW view_immotool_mails_parts AS
  SELECT * FROM immotool_mails_parts
  WHERE message_id IN (SELECT message_id FROM view_immotool_mails_messages);

GRANT SELECT
  ON view_immotool_mails_parts
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_mails_parts_content
-- -----------------------------------------------------
CREATE VIEW view_immotool_mails_parts_content AS
  SELECT
    part_id,
    part_file_content,
    created_at,
    modified_at
  FROM immotool_mails_parts
  WHERE message_id IN (SELECT message_id FROM view_immotool_mails_messages);

GRANT SELECT
  ON view_immotool_mails_parts_content
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_mails_headers
-- -----------------------------------------------------
CREATE VIEW view_immotool_mails_headers AS
  SELECT * FROM immotool_mails_headers
  WHERE part_id IN (SELECT part_id FROM view_immotool_mails_parts);

GRANT SELECT
  ON view_immotool_mails_headers
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_mails_addresses
-- -----------------------------------------------------
CREATE VIEW view_immotool_mails_addresses AS
  SELECT * FROM immotool_mails_addresses
  WHERE message_id IN (SELECT message_id FROM view_immotool_mails_messages);

GRANT SELECT
  ON view_immotool_mails_addresses
  TO "IMMOTOOL";
