-- -----------------------------------------------------
-- Agency-Addon for OpenEstate-ImmoTool
-- update 4 for HSQLDB
-- Copyright (C) 2009-2018 OpenEstate.org
-- -----------------------------------------------------
-- Add sequence 'seq_openestate_objects_links'
-- Add table 'openestate_objects_links'
-- Add table 'openestate_objects_links_i18n'
-- Add view 'view_openestate_objects_links'
-- Add view 'view_openestate_objects_links_i18n'
-- Add procedure 'remove_openestate_objects_links'
-- Add procedure 'remove_openestate_objects_links_i18n'
-- Add procedure 'save_openestate_objects_links'
-- Add procedure 'save_openestate_objects_links_i18n'
-- Update procedure 'remove_openestate_objects_files'
-- Update procedure 'remove_openestate_objects_files_i18n'
-- -----------------------------------------------------

-- -----------------------------------------------------
-- Sequence seq_openestate_objects_links
-- -----------------------------------------------------
CREATE SEQUENCE seq_openestate_objects_links
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;

-- -----------------------------------------------------
-- Table openestate_objects_links
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_links (
  link_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_objects_links NOT NULL,
  object_id BIGINT NOT NULL ,
  link_web_url VARCHAR(255) DEFAULT NULL NULL,
  link_provider_id VARCHAR(255) DEFAULT NULL NULL,
  link_provider_name VARCHAR(255) DEFAULT NULL NULL,
  link_published BOOLEAN DEFAULT TRUE NOT NULL,
  link_position INT DEFAULT 0 NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (link_id),
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_objects_links_i18n
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_links_i18n (
  link_id BIGINT NOT NULL ,
  language_id BIGINT NOT NULL ,
  i18n_title VARCHAR(255) DEFAULT NULL NULL,
  PRIMARY KEY (link_id, language_id),
  FOREIGN KEY (link_id)
    REFERENCES openestate_objects_links(link_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (language_id)
    REFERENCES openestate_languages(language_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

-- -----------------------------------------------------
-- View view_openestate_objects_links
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_links AS
  SELECT * FROM openestate_objects_links
  WHERE object_id IN (SELECT object_id FROM view_openestate_objects);

GRANT SELECT
  ON view_openestate_objects_links
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_links_i18n
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_links_i18n AS
  SELECT * FROM openestate_objects_links_i18n
  WHERE link_id IN (SELECT link_id FROM view_openestate_objects_links);

GRANT SELECT
  ON view_openestate_objects_links_i18n
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_links
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_links( val_link_id BIGINT )
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_link_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_links
      WHERE link_id = val_link_id;
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Datei entfernen
    DELETE
      FROM openestate_objects_links
      WHERE link_id = val_link_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_links
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_links_i18n
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_links_i18n( val_link_id BIGINT, val_language_id BIGINT )
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_link_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_links
      WHERE link_id = val_link_id;
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzungen zu einer Sprache entfernen
    IF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_links_i18n
        WHERE link_id = val_link_id
        AND language_id = val_language_id;

    -- Alle Übersetzungen entfernen
    ELSE

      DELETE
        FROM openestate_objects_links_i18n
        WHERE link_id = val_link_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_links_i18n
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_links
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_links(
  INOUT val_link_id BIGINT,
  val_object_id BIGINT,
  val_link_web_url VARCHAR(255),
  val_link_provider_id VARCHAR(255),
  val_link_provider_name VARCHAR(255),
  val_link_published BOOLEAN,
  val_link_position INT )
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuer Link
    IF val_link_id < 1 THEN

      -- Datei speichern
      INSERT INTO openestate_objects_links (
        object_id,
        link_web_url,
        link_provider_id,
        link_provider_name,
        link_published,
        link_position )
      VALUES (
        val_object_id,
        val_link_web_url,
        val_link_provider_id,
        val_link_provider_name,
        val_link_published,
        val_link_position );

      -- ID zurückliefern
      SET val_link_id = IDENTITY();

    ELSE

      -- Link bearbeiten
      UPDATE openestate_objects_links
        SET
          object_id = val_object_id,
          link_web_url = val_link_web_url,
          link_provider_id = val_link_provider_id,
          link_provider_name = val_link_provider_name,
          link_published = val_link_published,
          link_position = val_link_position,
          modified_at = NOW()
        WHERE
          link_id = val_link_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_links
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_openestate_objects_links_i18n
-- -----------------------------------------------------
CREATE PROCEDURE save_openestate_objects_links_i18n(
  val_link_id BIGINT,
  val_language_id BIGINT,
  val_i18n_title VARCHAR(255) )
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_link_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_links
      WHERE link_id = val_link_id;
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzung ggf. entfernen
    DELETE
      FROM openestate_objects_links_i18n
      WHERE link_id = val_link_id
      AND language_id = val_language_id;

    -- Übersetzung speichern
    INSERT INTO openestate_objects_links_i18n (
      link_id,
      language_id,
      i18n_title )
    VALUES (
      val_link_id,
      val_language_id,
      val_i18n_title );

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_links_i18n
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- Drop updated procedures
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects_files RESTRICT;
DROP PROCEDURE IF EXISTS remove_openestate_objects_files_i18n RESTRICT;

\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_files
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_files( val_file_id BIGINT )
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_files
      WHERE file_id = val_file_id;
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Datei entfernen
    DELETE
      FROM openestate_objects_files
      WHERE file_id = val_file_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_files
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_openestate_objects_files_i18n
-- -----------------------------------------------------
CREATE PROCEDURE remove_openestate_objects_files_i18n( val_file_id BIGINT, val_language_id BIGINT )
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_files
      WHERE file_id = val_file_id;
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzungen zu einer Sprache entfernen
    IF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_files_i18n
        WHERE file_id = val_file_id
        AND language_id = val_language_id;

    -- Alle Übersetzungen entfernen
    ELSE

      DELETE
        FROM openestate_objects_files_i18n
        WHERE file_id = val_file_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_files_i18n
  TO "IMMOTOOL";
