-- -----------------------------------------------------
-- Agency-Addon for OpenEstate-ImmoTool
-- schema for HSQLDB
-- Copyright (C) 2009-2018 OpenEstate.org
-- -----------------------------------------------------

CREATE SEQUENCE seq_immotool_agency_exposes
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_immotool_agency_exposes_files
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_immotool_agency_exports
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_immotool_agency_exports_logs
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;


-- -----------------------------------------------------
-- Table immotool_agency_objects
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_objects (
  object_id BIGINT NOT NULL,
  object_status VARCHAR(25) NOT NULL,
  object_export_mode VARCHAR(25) NOT NULL,
  object_import_id VARCHAR(255) DEFAULT NULL NULL,
  object_import_hash VARCHAR(255) DEFAULT NULL NULL,
  object_notes LONGVARCHAR DEFAULT NULL NULL,
  access_owner_id BIGINT DEFAULT 1 NOT NULL,
  access_group_id BIGINT DEFAULT 1 NOT NULL,
  access_permissions INT DEFAULT 0 NOT NULL,
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT
);


-- -----------------------------------------------------
-- Table immotool_agency_objects_logs
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_objects_logs (
  log_id BIGINT NOT NULL,
  object_id BIGINT NOT NULL,
  PRIMARY KEY (log_id, object_id),
  FOREIGN KEY (log_id)
    REFERENCES immotool_logs(log_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_agency_contacts
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_contacts (
  contact_id BIGINT NOT NULL,
  contact_enabled BOOLEAN DEFAULT true NOT NULL,
  contact_notes LONGVARCHAR DEFAULT NULL NULL,
  access_owner_id BIGINT DEFAULT 1 NOT NULL,
  access_group_id BIGINT DEFAULT 1 NOT NULL,
  access_permissions INT DEFAULT 0 NOT NULL,
  FOREIGN KEY (contact_id)
    REFERENCES openestate_contacts(contact_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT
);


-- -----------------------------------------------------
-- Table immotool_agency_exposes
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exposes (
  expose_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_agency_exposes NOT NULL,
  expose_name VARCHAR(100) NOT NULL,
  expose_type VARCHAR(255) NULL,
  access_owner_id BIGINT DEFAULT 1 NOT NULL,
  access_group_id BIGINT DEFAULT 1 NOT NULL,
  access_permissions INT DEFAULT 0 NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (expose_id),
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT
);


-- -----------------------------------------------------
-- Table immotool_agency_exposes_settings
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exposes_settings (
  expose_id BIGINT NOT NULL,
  setting_key VARCHAR(255) NOT NULL,
  setting_value LONGVARCHAR DEFAULT NULL NULL,
  PRIMARY KEY (expose_id, setting_key),
  FOREIGN KEY (expose_id)
    REFERENCES immotool_agency_exposes(expose_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_agency_expose_files
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exposes_files (
  file_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_agency_exposes_files NOT NULL,
  expose_id BIGINT NOT NULL,
  file_name VARCHAR(255) NOT NULL,
  file_mimetype VARCHAR(255) NOT NULL,
  file_content BLOB(1G) DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (file_id),
  FOREIGN KEY (expose_id)
    REFERENCES immotool_agency_exposes(expose_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_agency_searches
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_searches (
  search_id BIGINT NOT NULL ,
  search_type VARCHAR(10) DEFAULT 'TEMP' NOT NULL,
  search_object_status VARCHAR(255) DEFAULT NULL NULL,
  access_owner_id BIGINT DEFAULT 1 NOT NULL,
  access_group_id BIGINT DEFAULT 1 NOT NULL,
  access_permissions INT DEFAULT 0 NOT NULL,
  PRIMARY KEY (search_id),
  FOREIGN KEY (search_id)
    REFERENCES openestate_searches(search_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT
);


-- -----------------------------------------------------
-- Table immotool_agency_exports
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exports (
  export_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_agency_exports NOT NULL,
  search_id BIGINT DEFAULT NULL NULL,
  expose_id BIGINT DEFAULT NULL NULL,
  export_name VARCHAR(100) NOT NULL,
  export_enabled BOOLEAN DEFAULT true NOT NULL,
  export_format VARCHAR(255) NOT NULL,
  export_transport VARCHAR(255) NOT NULL,
  export_transfer VARCHAR(255) NOT NULL,
  export_limit INT DEFAULT 0 NOT NULL,
  export_last_execution TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  export_lock_time TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  export_notes LONGVARCHAR DEFAULT NULL NULL,
  access_owner_id BIGINT DEFAULT 1 NOT NULL,
  access_group_id BIGINT DEFAULT 1 NOT NULL,
  access_permissions INT DEFAULT 0 NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (export_id),
  FOREIGN KEY (search_id)
    REFERENCES immotool_agency_searches(search_id)
    MATCH SIMPLE
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (expose_id)
    REFERENCES immotool_agency_exposes(expose_id)
    MATCH SIMPLE
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT
);


-- -----------------------------------------------------
-- Table immotool_agency_exports_settings
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exports_settings (
  export_id BIGINT NOT NULL,
  setting_key VARCHAR(255) NOT NULL,
  setting_value LONGVARCHAR DEFAULT NULL NULL,
  PRIMARY KEY (export_id, setting_key),
  FOREIGN KEY (export_id)
    REFERENCES immotool_agency_exports(export_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_agency_exports_publishings
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exports_publishings (
  export_id BIGINT NOT NULL,
  publishing_object_id BIGINT NOT NULL,
  publishing_object_nr VARCHAR(255) DEFAULT NULL NULL,
  publishing_object_key VARCHAR(255) DEFAULT NULL NULL,
  publishing_hash VARCHAR(40) DEFAULT NULL NULL,
  publishing_time TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (export_id, publishing_object_id),
  FOREIGN KEY (export_id)
    REFERENCES immotool_agency_exports(export_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_agency_exports_logs
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exports_logs (
  log_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_agency_exports_logs NOT NULL,
  export_id BIGINT NOT NULL,
  log_format VARCHAR(100) NOT NULL,
  log_transport VARCHAR(100) NOT NULL,
  log_transfer VARCHAR(100) NOT NULL,
  log_limit INT DEFAULT 0 NOT NULL,
  log_count_inserts INT DEFAULT 0 NOT NULL,
  log_count_updates INT DEFAULT 0 NOT NULL,
  log_count_deletes INT DEFAULT 0 NOT NULL,
  log_count_errors INT DEFAULT 0 NOT NULL,
  log_count_warnings INT DEFAULT 0 NOT NULL,
  log_count_notices INT DEFAULT 0 NOT NULL,
  log_time TIMESTAMP WITH TIME ZONE NOT NULL,
  PRIMARY KEY (log_id),
  FOREIGN KEY (export_id)
    REFERENCES immotool_agency_exports(export_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_agency_exports_logs_entries
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exports_logs_entries (
  log_id BIGINT NOT NULL,
  entry_object_id BIGINT NOT NULL,
  entry_object_nr VARCHAR(255) DEFAULT NULL NULL,
  entry_action VARCHAR(10) NOT NULL,
  entry_exported BOOLEAN DEFAULT false NOT NULL,
  PRIMARY KEY (log_id, entry_object_id),
  FOREIGN KEY (log_id)
    REFERENCES immotool_agency_exports_logs(log_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_agency_exports_logs_messages
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_exports_logs_messages (
  log_id BIGINT NOT NULL,
  entry_object_id BIGINT NOT NULL,
  message_level VARCHAR(10) NOT NULL,
  message_code VARCHAR(255) DEFAULT NULL NULL,
  message_text LONGVARCHAR NOT NULL,
  FOREIGN KEY (log_id)
    REFERENCES immotool_agency_exports_logs(log_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table immotool_agency_objects_exports
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_agency_objects_exports (
  object_id BIGINT NOT NULL,
  export_id BIGINT NOT NULL,
  PRIMARY KEY (object_id, export_id),
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (export_id)
    REFERENCES immotool_agency_exports(export_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- View view_immotool_agency_exports
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exports AS
  SELECT * FROM immotool_agency_exports
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(access_permissions, 64) = 64)
    OR
    (BITAND(access_permissions, 8) = 8 AND access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(access_permissions, 1) = 1 AND access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_immotool_agency_exports
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exports_logs
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exports_logs AS
  SELECT * FROM immotool_agency_exports_logs
  WHERE export_id IN (SELECT export_id FROM view_immotool_agency_exports);

GRANT SELECT
  ON view_immotool_agency_exports_logs
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exports_logs_entries
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exports_logs_entries AS
  SELECT * FROM immotool_agency_exports_logs_entries
  WHERE log_id IN (SELECT log_id FROM view_immotool_agency_exports_logs);

GRANT SELECT
  ON view_immotool_agency_exports_logs_entries
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exports_logs_messages
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exports_logs_messages AS
  SELECT * FROM immotool_agency_exports_logs_messages
  WHERE log_id IN (SELECT log_id FROM view_immotool_agency_exports_logs);

GRANT SELECT
  ON view_immotool_agency_exports_logs_messages
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exports_publishings
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exports_publishings AS
  SELECT * FROM immotool_agency_exports_publishings
  WHERE export_id IN (SELECT export_id FROM view_immotool_agency_exports);

GRANT SELECT
  ON view_immotool_agency_exports_publishings
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exports_settings
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exports_settings AS
  SELECT * FROM immotool_agency_exports_settings
  WHERE export_id IN (SELECT export_id FROM view_immotool_agency_exports);

GRANT SELECT
  ON view_immotool_agency_exports_settings
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exposes
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exposes AS
  SELECT * FROM immotool_agency_exposes
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(access_permissions, 64) = 64)
    OR
    (BITAND(access_permissions, 8) = 8 AND access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(access_permissions, 1) = 1 AND access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_immotool_agency_exposes
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exposes_files
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exposes_files AS
  SELECT
    file_id,
    expose_id,
    file_name,
    file_mimetype,
    created_at,
    modified_at
  FROM immotool_agency_exposes_files
  WHERE expose_id IN (SELECT expose_id FROM view_immotool_agency_exposes);

GRANT SELECT
  ON view_immotool_agency_exposes_files
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exposes_files_content
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exposes_files_content AS
  SELECT
    file_id,
    file_content,
    created_at,
    modified_at
  FROM immotool_agency_exposes_files
  WHERE expose_id IN (SELECT expose_id FROM view_immotool_agency_exposes);

GRANT SELECT
  ON view_immotool_agency_exposes_files_content
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_exposes_settings
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_exposes_settings AS
  SELECT * FROM immotool_agency_exposes_settings
  WHERE expose_id IN (SELECT expose_id FROM view_immotool_agency_exposes);

GRANT SELECT
  ON view_immotool_agency_exposes_settings
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_objects_exports
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_objects_exports AS
  SELECT * FROM immotool_agency_objects_exports
  WHERE export_id IN (SELECT export_id FROM view_immotool_agency_exports);

GRANT SELECT
  ON view_immotool_agency_objects_exports
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_immotool_agency_objects_logs
-- -----------------------------------------------------
CREATE VIEW view_immotool_agency_objects_logs AS
  SELECT a.object_id, b.* FROM immotool_agency_objects_logs a
  INNER JOIN immotool_logs b ON ( a.log_id = b.log_id )
  WHERE a.object_id IN (SELECT object_id FROM view_openestate_objects)
  AND b.plugin_id IN (SELECT plugin_id FROM view_immotool_plugins WHERE plugin_name='OpenEstate-Tool-Agency');

GRANT SELECT
  ON view_immotool_agency_objects_logs
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_contacts
-- -----------------------------------------------------
ALTER VIEW view_openestate_contacts AS
  SELECT
    a.*,
    b.contact_enabled,
    b.contact_notes,
    b.access_owner_id,
    b.access_group_id,
    b.access_permissions
  FROM openestate_contacts a
  INNER JOIN immotool_agency_contacts b ON ( a.contact_id = b.contact_id )
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(b.access_permissions, 64) = 64)
    OR
    (BITAND(b.access_permissions, 8) = 8 AND b.access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(b.access_permissions, 1) = 1 AND b.access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_openestate_contacts
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects AS
  SELECT
    a.*,
    b.object_status,
    b.object_export_mode,
    b.object_import_id,
    b.object_import_hash,
    b.object_notes,
    b.access_owner_id,
    b.access_group_id,
    b.access_permissions
  FROM openestate_objects a
  INNER JOIN immotool_agency_objects b ON ( a.object_id = b.object_id )
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(b.access_permissions, 64) = 64)
    OR
    (BITAND(b.access_permissions, 8) = 8 AND b.access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(b.access_permissions, 1) = 1 AND b.access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_openestate_objects
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_actions
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_actions AS
  SELECT object_action FROM view_openestate_objects GROUP BY object_action;

GRANT SELECT
  ON view_openestate_objects_actions
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_attribs
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_attribs AS
  SELECT * FROM openestate_objects_attribs
  WHERE object_id IN (SELECT object_id FROM view_openestate_objects);

GRANT SELECT
  ON view_openestate_objects_attribs
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_files
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_files AS
  SELECT
    file_id,
    object_id,
    file_name,
    file_url,
    file_mimetype,
    file_published,
    file_type,
    file_size,
    file_hash,
    file_image_width,
    file_image_height,
    file_position,
    created_at,
    modified_at
  FROM openestate_objects_files
  WHERE object_id IN (SELECT object_id FROM view_openestate_objects);

GRANT SELECT
  ON view_openestate_objects_files
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_files_content
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_files_content AS
  SELECT
    file_id,
    file_content,
    created_at,
    modified_at
  FROM openestate_objects_files
  WHERE object_id IN (SELECT object_id FROM view_openestate_objects);

GRANT SELECT
  ON view_openestate_objects_files_content
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_files_preview
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_files_preview AS
  SELECT
    file_id,
    file_content_preview,
    created_at,
    modified_at
  FROM openestate_objects_files
  WHERE object_id IN (SELECT object_id FROM view_openestate_objects);

GRANT SELECT
  ON view_openestate_objects_files_preview
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_files_i18n
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_files_i18n AS
  SELECT * FROM openestate_objects_files_i18n
  WHERE file_id IN (SELECT file_id FROM view_openestate_objects_files);

GRANT SELECT
  ON view_openestate_objects_files_i18n
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_i18n
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_i18n AS
  SELECT * FROM openestate_objects_i18n
  WHERE object_id IN (SELECT object_id FROM view_openestate_objects);

GRANT SELECT
  ON view_openestate_objects_i18n
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_links
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_links AS
  SELECT * FROM openestate_objects_links
  WHERE object_id IN (SELECT object_id FROM view_openestate_objects);

GRANT SELECT
  ON view_openestate_objects_links
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_links_i18n
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_links_i18n AS
  SELECT * FROM openestate_objects_links_i18n
  WHERE link_id IN (SELECT link_id FROM view_openestate_objects_links);

GRANT SELECT
  ON view_openestate_objects_links_i18n
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_objects_types
-- -----------------------------------------------------
ALTER VIEW view_openestate_objects_types AS
  SELECT object_type FROM view_openestate_objects GROUP BY object_type;

GRANT SELECT
  ON view_openestate_objects_types
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_searches
-- -----------------------------------------------------
ALTER VIEW view_openestate_searches AS
  SELECT
    a.*,
    b.search_type,
    b.search_object_status,
    b.access_owner_id,
    b.access_group_id,
    b.access_permissions
  FROM openestate_searches a
  INNER JOIN immotool_agency_searches b ON ( a.search_id = b.search_id )
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(b.access_permissions, 64) = 64)
    OR
    (BITAND(b.access_permissions, 8) = 8 AND b.access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(b.access_permissions, 1) = 1 AND b.access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_openestate_searches
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_searches_attribs
-- -----------------------------------------------------
ALTER VIEW view_openestate_searches_attribs AS
  SELECT * FROM openestate_searches_attribs
  WHERE search_id IN (SELECT search_id FROM view_openestate_searches);

GRANT SELECT
  ON view_openestate_searches_attribs
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_searches_params
-- -----------------------------------------------------
ALTER VIEW view_openestate_searches_params AS
  SELECT * FROM openestate_searches_params
  WHERE search_id IN (SELECT search_id FROM view_openestate_searches);

GRANT SELECT
  ON view_openestate_searches_params
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- View view_openestate_searches_values
-- -----------------------------------------------------
ALTER VIEW view_openestate_searches_values AS
  SELECT * FROM openestate_searches_values
  WHERE search_id IN (SELECT search_id FROM view_openestate_searches);

GRANT SELECT
  ON view_openestate_searches_values
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- Further Permissions
-- -----------------------------------------------------
GRANT SELECT
  ON view_openestate_geodbs
  TO "IMMOTOOL";
GRANT SELECT
  ON view_openestate_geodbs_areas
  TO "IMMOTOOL";
GRANT SELECT
  ON view_openestate_geodbs_countries
  TO "IMMOTOOL";
GRANT SELECT
  ON view_openestate_languages
  TO "IMMOTOOL";
GRANT SELECT
  ON view_openestate_objects_geodbs_areas
  TO "IMMOTOOL";
