-- -----------------------------------------------------
-- Contacts extensions to the Agency-Addon for OpenEstate-ImmoTool
-- update 2 for HSQLDB
-- Copyright (C) 2009-2018 OpenEstate.org
-- -----------------------------------------------------
-- Add a specific name to all procedures & functions
-- Add schema and routines for agency brokerings
-- Add schema and routines for agency interests
-- Read agency brokerings from 'immotool_agency_objects_contacts_entries'
-- Read agency interests from 'immotool_agency_searches_contacts_entries'
-- Drop table, view & routines for 'immotool_agency_searches_contacts_entries'
-- -----------------------------------------------------

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_objects_contacts_entries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_objects_contacts_entries CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_objects_contacts_entries(
    val_agency_object_id BIGINT,
    val_contacts_entry_id BIGINT
  )
  SPECIFIC remove_immotool_agency_objects_contacts_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    IF val_agency_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF val_contacts_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lese-Rechte besitzt
    IF NOT is_admin() = TRUE THEN

      -- Lese-Rechte auf der Immobilie müssen vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_agency_objects
        WHERE object_id = val_agency_object_id;
      IF NOT can_read( permissions, owner_uid, owner_gid ) = TRUE THEN
        SIGNAL SQLSTATE '45000';
      END IF;

      -- Lese-Rechte auf dem Adresseintrag müssen vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_contacts_entries
        WHERE entry_id = val_contacts_entry_id;
      IF NOT can_read( permissions, owner_uid, owner_gid ) = TRUE THEN
        SIGNAL SQLSTATE '45000';
      END IF;
    END IF;

    -- Relation entfernen
    DELETE
      FROM immotool_agency_objects_contacts_entries
      WHERE agency_object_id = val_agency_object_id
      AND contacts_entry_id = val_contacts_entry_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_objects_contacts_entries
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_objects_contacts_entries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_objects_contacts_entries CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_objects_contacts_entries(
    val_agency_object_id BIGINT,
    val_contacts_entry_id BIGINT,
    val_relation_type VARCHAR(100),
    val_relation_notes LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_objects_contacts_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Relation ggf. entfernen
    CALL remove_immotool_agency_objects_contacts_entries( val_agency_object_id, val_contacts_entry_id );

    -- Relation einfügen
    INSERT INTO immotool_agency_objects_contacts_entries (
      agency_object_id,
      contacts_entry_id,
      relation_type,
      relation_notes )
    VALUES (
      val_agency_object_id,
      val_contacts_entry_id,
      val_relation_type,
      val_relation_notes );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_objects_contacts_entries
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Schema for agency brokerings
-- -----------------------------------------------------
CREATE SEQUENCE seq_immotool_agency_brokerings
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;

CREATE CACHED TABLE immotool_agency_brokerings (
  brokering_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_agency_brokerings NOT NULL,
  contacts_entry_id BIGINT NOT NULL,
  agency_object_id BIGINT NOT NULL,
  brokering_exclusive BOOLEAN DEFAULT FALSE NOT NULL,
  brokering_begin TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  brokering_end TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  brokering_notes LONGVARCHAR DEFAULT NULL NULL,
  access_owner_id BIGINT DEFAULT NULL NULL,
  access_group_id BIGINT DEFAULT NULL NULL,
  access_permissions INT DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (brokering_id),
  UNIQUE (agency_object_id),
  FOREIGN KEY (contacts_entry_id)
    REFERENCES immotool_contacts_entries(entry_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (agency_object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

CREATE VIEW view_immotool_agency_brokerings AS
  SELECT * FROM immotool_agency_brokerings
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(access_permissions, 64) = 64)
    OR
    (BITAND(access_permissions, 8) = 8 AND access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(access_permissions, 1) = 1 AND access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_immotool_agency_brokerings
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Schema for agency interests
-- -----------------------------------------------------
CREATE SEQUENCE seq_immotool_agency_interests
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;

CREATE CACHED TABLE immotool_agency_interests (
  interest_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_agency_interests NOT NULL,
  contacts_entry_id BIGINT NOT NULL,
  agency_search_id BIGINT NOT NULL,
  interest_status VARCHAR(100) NULL,
  interest_begin TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  interest_end TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  interest_notes LONGVARCHAR DEFAULT NULL NULL,
  access_owner_id BIGINT DEFAULT NULL NULL,
  access_group_id BIGINT DEFAULT NULL NULL,
  access_permissions INT DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (interest_id),
  FOREIGN KEY (contacts_entry_id)
    REFERENCES immotool_contacts_entries(entry_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (agency_search_id)
    REFERENCES immotool_agency_searches(search_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

CREATE VIEW view_immotool_agency_interests AS
  SELECT * FROM immotool_agency_interests
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(access_permissions, 64) = 64)
    OR
    (BITAND(access_permissions, 8) = 8 AND access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(access_permissions, 1) = 1 AND access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_immotool_agency_interests
  TO "IMMOTOOL";

CREATE CACHED TABLE immotool_agency_interests_offers (
  interest_id BIGINT NOT NULL,
  agency_object_id BIGINT NOT NULL,
  offer_time TIMESTAMP WITH TIME ZONE NOT NULL,
  offer_status VARCHAR(100) NOT NULL,
  offer_notes LONGVARCHAR DEFAULT NULL NULL,
  PRIMARY KEY (interest_id, agency_object_id),
  FOREIGN KEY (interest_id)
    REFERENCES immotool_agency_interests(interest_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (agency_object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

CREATE VIEW view_immotool_agency_interests_offers AS
  SELECT * FROM immotool_agency_interests_offers
  WHERE interest_id IN (SELECT interest_id FROM view_immotool_agency_interests)
  AND agency_object_id IN (SELECT object_id FROM view_openestate_objects);

GRANT SELECT
  ON view_immotool_agency_interests_offers
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Function can_delete_immotool_agency_brokerings
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_agency_brokerings( val_brokering_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_brokerings
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_brokerings
      WHERE brokering_id = val_brokering_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_agency_interests
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_agency_interests( val_interest_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_interests
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_interests
      WHERE interest_id = val_interest_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_agency_brokerings
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_agency_brokerings( val_brokering_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_brokerings
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_brokerings
      WHERE brokering_id = val_brokering_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_agency_interests
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_agency_interests( val_interest_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_interests
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_interests
      WHERE interest_id = val_interest_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_agency_brokerings
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_agency_brokerings( val_brokering_id BIGINT )
  SPECIFIC remove_immotool_agency_brokerings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_brokering_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_delete_immotool_agency_brokerings( val_brokering_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag entfernen
    DELETE
      FROM immotool_agency_brokerings
      WHERE brokering_id = val_brokering_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_brokerings
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_agency_interests
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_agency_interests( val_interest_id BIGINT )
  SPECIFIC remove_immotool_agency_interests
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_interest_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_delete_immotool_agency_interests( val_interest_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag entfernen
    DELETE
      FROM immotool_agency_interests
      WHERE interest_id = val_interest_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_interests
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_agency_interests_offers
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_agency_interests_offers(
    val_interest_id BIGINT,
    val_agency_object_id BIGINT
  )
  SPECIFIC remove_immotool_agency_interests_offers
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_interest_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_interests(val_interest_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF val_agency_object_id < 1 THEN

      -- alle Relationen entfernen
      DELETE
        FROM immotool_agency_interests_offers
        WHERE interest_id = val_interest_id;

    ELSE

      -- Relation entfernen
      DELETE
        FROM immotool_agency_interests_offers
        WHERE interest_id = val_interest_id
        AND agency_object_id = val_agency_object_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_interests_offers
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_agency_brokerings
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_agency_brokerings(
    INOUT val_brokering_id BIGINT,
    val_contacts_entry_id BIGINT,
    val_agency_object_id BIGINT,
    val_brokering_exclusive BOOLEAN,
    val_brokering_begin TIMESTAMP WITH TIME ZONE,
    val_brokering_end TIMESTAMP WITH TIME ZONE,
    val_brokering_notes LONGVARCHAR,
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_agency_brokerings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Neue Vermarktung erzeugen
    IF val_brokering_id < 1 THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Vermarktung speichern
      INSERT INTO immotool_agency_brokerings (
        contacts_entry_id,
        agency_object_id,
        brokering_exclusive,
        brokering_begin,
        brokering_end,
        brokering_notes,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_contacts_entry_id,
        val_agency_object_id,
        val_brokering_exclusive,
        val_brokering_begin,
        val_brokering_end,
        val_brokering_notes,
        owner_uid,
        owner_gid,
        permissions);

      -- ID zurückliefern
      SET val_brokering_id = IDENTITY();

    -- Bestehende Vermarktung bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_agency_brokerings
        WHERE brokering_id = val_brokering_id;

      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>1 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>1 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>1 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Vermarktung bearbeiten
      UPDATE immotool_agency_brokerings
        SET
          contacts_entry_id = val_contacts_entry_id,
          agency_object_id = val_agency_object_id,
          brokering_exclusive = val_brokering_exclusive,
          brokering_begin = val_brokering_begin,
          brokering_end = val_brokering_end,
          brokering_notes = val_brokering_notes,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions,
          modified_at = NOW()
        WHERE
          brokering_id = val_brokering_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_brokerings
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_agency_interests
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_agency_interests(
    INOUT val_interest_id BIGINT,
    val_contacts_entry_id BIGINT,
    val_agency_search_id BIGINT,
    val_interest_status VARCHAR(100),
    val_interest_begin TIMESTAMP WITH TIME ZONE,
    val_interest_end TIMESTAMP WITH TIME ZONE,
    val_interest_notes LONGVARCHAR,
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_agency_interests
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Neuen Interessent erzeugen
    IF val_interest_id < 1 THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Interessent speichern
      INSERT INTO immotool_agency_interests (
        contacts_entry_id,
        agency_search_id,
        interest_status,
        interest_begin,
        interest_end,
        interest_notes,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_contacts_entry_id,
        val_agency_search_id,
        val_interest_status,
        val_interest_begin,
        val_interest_end,
        val_interest_notes,
        owner_uid,
        owner_gid,
        permissions);

      -- ID zurückliefern
      SET val_interest_id = IDENTITY();

    -- Bestehenden Interessent bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_agency_interests
        WHERE interest_id = val_interest_id;

      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>1 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>1 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>1 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Interessent bearbeiten
      UPDATE immotool_agency_interests
        SET
          contacts_entry_id = val_contacts_entry_id,
          agency_search_id = val_agency_search_id,
          interest_status = val_interest_status,
          interest_begin = val_interest_begin,
          interest_end = val_interest_end,
          interest_notes = val_interest_notes,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions,
          modified_at = NOW()
        WHERE
          interest_id = val_interest_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_interests
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_agency_interests_offers
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_agency_interests_offers(
    val_interest_id BIGINT,
    val_agency_object_id BIGINT,
    val_offer_time TIMESTAMP WITH TIME ZONE,
    val_offer_status VARCHAR(100),
    val_offer_notes LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_interests_offers
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_interest_id < 1 OR val_agency_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Relation ggf. entfernen
    CALL remove_immotool_agency_interests_offers( val_interest_id, val_agency_object_id );

    -- Relation einfügen
    INSERT INTO immotool_agency_interests_offers (
      interest_id,
      agency_object_id,
      offer_time,
      offer_status,
      offer_notes )
    VALUES (
      val_interest_id,
      val_agency_object_id,
      val_offer_time,
      val_offer_status,
      val_offer_notes );

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_interests_offers
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Read agency brokerings from 'immotool_agency_objects_contacts_entries'
-- -----------------------------------------------------
INSERT INTO immotool_agency_brokerings
(
  contacts_entry_id,
  agency_object_id,
  brokering_begin,
  brokering_notes,
  access_owner_id,
  access_group_id,
  access_permissions,
  created_at,
  modified_at
)
(
  SELECT
    a.contacts_entry_id,
    a.agency_object_id,
    a.created_at,
    a.relation_notes,
    b.access_owner_id,
    b.access_group_id,
    b.access_permissions,
    a.created_at,
    CURRENT_TIMESTAMP
  FROM immotool_agency_objects_contacts_entries a
  INNER JOIN immotool_agency_objects b ON a.agency_object_id=b.object_id
  WHERE (a.agency_object_id, a.contacts_entry_id) IN
  (
    SELECT agency_object_id, MIN(contacts_entry_id)
    FROM immotool_agency_objects_contacts_entries
    WHERE relation_type IN ('owner', 'Eigentümer')
    GROUP BY agency_object_id
  )
);
DELETE FROM immotool_agency_objects_contacts_entries
  WHERE relation_type IN ('owner', 'Eigentümer')
  AND (agency_object_id, contacts_entry_id) IN (SELECT agency_object_id, contacts_entry_id FROM immotool_agency_brokerings);


-- -----------------------------------------------------
-- Read agency interests from 'immotool_agency_searches_contacts_entries'
-- -----------------------------------------------------
INSERT INTO immotool_agency_interests
(
  contacts_entry_id,
  agency_search_id,
  interest_status,
  interest_begin,
  interest_notes,
  access_owner_id,
  access_group_id,
  access_permissions,
  created_at,
  modified_at
)
(
  SELECT
    a.contacts_entry_id,
    a.agency_search_id,
    'PROCESSING',
    a.created_at,
    a.relation_notes,
    b.access_owner_id,
    b.access_group_id,
    b.access_permissions,
    a.created_at,
    a.modified_at
  FROM immotool_agency_searches_contacts_entries a
  INNER JOIN immotool_agency_searches b ON a.agency_search_id=b.search_id
  INNER JOIN openestate_searches c ON a.agency_search_id=c.search_id
  WHERE c.search_enabled = TRUE
);
INSERT INTO immotool_agency_interests
(
  contacts_entry_id,
  agency_search_id,
  interest_status,
  interest_begin,
  interest_notes,
  access_owner_id,
  access_group_id,
  access_permissions,
  created_at,
  modified_at
)
(
  SELECT
    a.contacts_entry_id,
    a.agency_search_id,
    'CANCELLED',
    a.created_at,
    a.relation_notes,
    b.access_owner_id,
    b.access_group_id,
    b.access_permissions,
    a.created_at,
    a.modified_at
  FROM immotool_agency_searches_contacts_entries a
  INNER JOIN immotool_agency_searches b ON a.agency_search_id=b.search_id
  INNER JOIN openestate_searches c ON a.agency_search_id=c.search_id
  WHERE c.search_enabled = FALSE
);
UPDATE immotool_agency_searches
  SET search_sticky = FALSE
  WHERE search_id IN (SELECT agency_search_id FROM immotool_agency_interests);
UPDATE immotool_agency_searches
  SET search_sticky = TRUE
  WHERE search_id NOT IN (SELECT agency_search_id FROM immotool_agency_interests);
UPDATE openestate_searches
  SET search_enabled = FALSE
  WHERE search_id IN (SELECT agency_search_id FROM immotool_agency_interests WHERE interest_status NOT IN ('PROCESSING', 'PLANNED'));
UPDATE openestate_searches
  SET search_enabled = TRUE
  WHERE search_id IN (SELECT agency_search_id FROM immotool_agency_interests WHERE interest_status IN ('PROCESSING', 'PLANNED'));
UPDATE openestate_searches
  SET search_enabled = TRUE
  WHERE search_id NOT IN (SELECT agency_search_id FROM immotool_agency_interests);

-- -----------------------------------------------------
-- Drop table, view & routines for 'immotool_agency_searches_contacts_entries'
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_searches_contacts_entries CASCADE;
DROP PROCEDURE IF EXISTS save_immotool_agency_searches_contacts_entries CASCADE;
DROP VIEW IF EXISTS view_immotool_agency_searches_contacts_entries CASCADE;
DROP TABLE IF EXISTS immotool_agency_searches_contacts_entries CASCADE;
