-- -----------------------------------------------------
-- Contacts-Addon for OpenEstate-ImmoTool
-- schema for HSQLDB
-- Copyright (C) 2009-2018 OpenEstate.org
-- -----------------------------------------------------

CREATE SEQUENCE seq_immotool_contacts_groups
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_immotool_contacts_entries
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_immotool_contacts_entries_files
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;


-- -----------------------------------------------------
-- Table immotool_contacts_groups
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_contacts_groups (
  group_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_contacts_groups NOT NULL,
  group_name VARCHAR(100) NOT NULL,
  group_notes LONGVARCHAR DEFAULT NULL NULL,
  access_owner_id BIGINT DEFAULT 1 NOT NULL,
  access_group_id BIGINT DEFAULT 1 NOT NULL,
  access_permissions INT DEFAULT 0 NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (group_id),
  FOREIGN KEY (access_owner_id)
    REFERENCES immotool_users(user_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT,
  FOREIGN KEY (access_group_id)
    REFERENCES immotool_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE SET DEFAULT
);

CREATE VIEW view_immotool_contacts_groups AS
  SELECT * FROM immotool_contacts_groups
  WHERE
  (
    ('DBA' IN (SELECT authorization_name FROM information_schema.authorizations WHERE authorization_type = 'ROLE'))
    OR
    (BITAND(access_permissions, 64) = 64)
    OR
    (BITAND(access_permissions, 8) = 8 AND access_group_id IN (SELECT group_id FROM view_immotool_users_groups WHERE user_login=USER()))
    OR
    (BITAND(access_permissions, 1) = 1 AND access_owner_id IN (SELECT user_id FROM view_immotool_users WHERE user_login=USER()))
  );

GRANT SELECT
  ON view_immotool_contacts_groups
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- Table immotool_contacts_entries
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_contacts_entries (
  entry_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_contacts_entries NOT NULL,
  group_id BIGINT NOT NULL,
  entry_public_id VARCHAR(255) DEFAULT NULL NULL,
  entry_notes LONGVARCHAR DEFAULT NULL NULL,
  entry_active BOOLEAN DEFAULT true NOT NULL,
  entry_birthday DATE DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (entry_id),
  UNIQUE (entry_public_id),
  FOREIGN KEY (group_id)
    REFERENCES immotool_contacts_groups(group_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

CREATE VIEW view_immotool_contacts_entries AS
  SELECT * FROM immotool_contacts_entries
  WHERE group_id IN (SELECT group_id FROM view_immotool_contacts_groups);

GRANT SELECT
  ON view_immotool_contacts_entries
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- Table immotool_contacts_entries_logs
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_contacts_entries_logs (
  log_id BIGINT NOT NULL,
  entry_id BIGINT NOT NULL,
  PRIMARY KEY (log_id, entry_id),
  FOREIGN KEY (log_id)
    REFERENCES immotool_logs(log_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (entry_id)
    REFERENCES immotool_contacts_entries(entry_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

CREATE VIEW view_immotool_contacts_entries_logs AS
  SELECT a.entry_id, b.* FROM immotool_contacts_entries_logs a
  INNER JOIN immotool_logs b ON ( a.log_id = b.log_id )
  WHERE a.entry_id IN (SELECT entry_id FROM view_immotool_contacts_entries)
  AND b.plugin_id IN (SELECT plugin_id FROM view_immotool_plugins WHERE plugin_name='OpenEstate-Tool-Contacts');

GRANT SELECT
  ON view_immotool_contacts_entries_logs
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- Table immotool_contacts_entries_settings
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_contacts_entries_settings (
  entry_id BIGINT NOT NULL,
  setting_key VARCHAR(255) NOT NULL,
  setting_value LONGVARCHAR DEFAULT NULL NULL,
  PRIMARY KEY (entry_id, setting_key),
  FOREIGN KEY (entry_id)
    REFERENCES immotool_contacts_entries(entry_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

CREATE VIEW view_immotool_contacts_entries_settings AS
  SELECT * FROM immotool_contacts_entries_settings
  WHERE entry_id IN (SELECT entry_id FROM view_immotool_contacts_entries);

GRANT SELECT
  ON view_immotool_contacts_entries_settings
  TO "IMMOTOOL";


-- -----------------------------------------------------
-- Table immotool_contacts_entries_files
-- -----------------------------------------------------
CREATE CACHED TABLE immotool_contacts_entries_files (
  file_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_immotool_contacts_entries_files NOT NULL,
  entry_id BIGINT NOT NULL,
  file_name VARCHAR(255) NOT NULL,
  file_mimetype VARCHAR(255) NULL,
  file_content BLOB(1G) DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (file_id),
  FOREIGN KEY (entry_id)
    REFERENCES immotool_contacts_entries(entry_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

CREATE VIEW view_immotool_contacts_entries_files (
  file_id,
  entry_id,
  file_name,
  file_mimetype,
  created_at,
  modified_at) AS
  SELECT
    file_id,
    entry_id,
    file_name,
    file_mimetype,
    created_at,
    modified_at
  FROM immotool_contacts_entries_files
  WHERE entry_id IN (SELECT entry_id FROM view_immotool_contacts_entries);

GRANT SELECT
  ON view_immotool_contacts_entries_files
  TO "IMMOTOOL";

CREATE VIEW view_immotool_contacts_entries_files_content (
  file_id,
  file_content,
  created_at,
  modified_at) AS
  SELECT
    file_id,
    file_content,
    created_at,
    modified_at
  FROM immotool_contacts_entries_files
  WHERE entry_id IN (SELECT entry_id FROM view_immotool_contacts_entries);

GRANT SELECT
  ON view_immotool_contacts_entries_files_content
  TO "IMMOTOOL";
