-- -----------------------------------------------------
-- News-Addon for OpenEstate-ImmoTool
-- update 3 for HSQLDB
-- Copyright (C) 2009-2018 OpenEstate.org
-- -----------------------------------------------------
-- Disable haufe.de RSS-Feed
-- Add a specific name to all procedures & functions
-- Bugfix within the can_* functions
-- -----------------------------------------------------

-- -----------------------------------------------------
-- Disable haufe.de RSS-Feed
-- -----------------------------------------------------
UPDATE immotool_news_feeds
  SET feed_update_enabled = FALSE
  WHERE feed_url = 'http://www.haufe.de/immobilien/rssNews';

-- -----------------------------------------------------
-- Function can_delete_immotool_news_feeds
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_delete_immotool_news_feeds CASCADE;
\.
CREATE FUNCTION can_delete_immotool_news_feeds( val_feed_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_news_feeds
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_news_feeds
      WHERE feed_id = val_feed_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_write_immotool_news_feeds
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_write_immotool_news_feeds CASCADE;
\.
CREATE FUNCTION can_write_immotool_news_feeds( val_feed_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_news_feeds
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_news_feeds
      WHERE feed_id = val_feed_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Procedure remove_immotool_news_entries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_news_entries CASCADE;
\.
CREATE PROCEDURE remove_immotool_news_entries( val_entry_id BIGINT )
  SPECIFIC remove_immotool_news_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_feed_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT feed_id
      INTO val_feed_id
      FROM immotool_news_entries
      WHERE entry_id = val_entry_id;
    SET allowed = can_write_immotool_news_feeds( val_feed_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag entfernen
    DELETE
      FROM immotool_news_entries
      WHERE entry_id = val_entry_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_news_entries
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_news_feeds
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_news_feeds CASCADE;
\.
CREATE PROCEDURE remove_immotool_news_feeds( val_feed_id BIGINT )
  SPECIFIC remove_immotool_news_feeds
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_feed_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_news_feeds( val_feed_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Feed entfernen
    DELETE
      FROM immotool_news_feeds
      WHERE feed_id = val_feed_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_news_feeds
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_news_entries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_news_entries CASCADE;
\.
CREATE PROCEDURE save_immotool_news_entries(
    INOUT val_entry_id BIGINT,
    val_feed_id BIGINT,
    val_entry_uri VARCHAR(255),
    val_entry_title VARCHAR(255),
    val_entry_author VARCHAR(255),
    val_entry_content LONGVARCHAR,
    val_entry_link VARCHAR(255),
    val_entry_time TIMESTAMP WITH TIME ZONE,
    val_entry_new BOOLEAN,
    val_entry_sticky BOOLEAN
  )
  SPECIFIC save_immotool_news_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_feed_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_feed_id IS NOT NULL AND val_feed_id > 0 THEN
      SET allowed = can_write_immotool_news_feeds( val_feed_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuen Eintrag erzeugen
    IF val_entry_id < 1 THEN
      -- Eintrag speichern
      INSERT INTO immotool_news_entries (
        feed_id,
        entry_uri,
        entry_title,
        entry_author,
        entry_content,
        entry_link,
        entry_time,
        entry_new,
        entry_sticky )
      VALUES (
        val_feed_id,
        val_entry_uri,
        val_entry_title,
        val_entry_author,
        val_entry_content,
        val_entry_link,
        val_entry_time,
        val_entry_new,
        val_entry_sticky );

      -- ID zurückliefern
      SET val_entry_id = IDENTITY();

    -- Bestehenden Eintrag bearbeiten
    ELSE
      -- Eintrag bearbeiten
      UPDATE immotool_news_entries
        SET
          feed_id = val_feed_id,
          entry_uri = val_entry_uri,
          entry_title = val_entry_title,
          entry_author = val_entry_author,
          entry_content = val_entry_content,
          entry_link = val_entry_link,
          entry_time = val_entry_time,
          entry_new = val_entry_new,
          entry_sticky = val_entry_sticky,
          modified_at = NOW()
        WHERE
          entry_id = val_entry_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_news_entries
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_news_feeds
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_news_feeds CASCADE;
\.
CREATE PROCEDURE save_immotool_news_feeds(
    INOUT val_feed_id BIGINT,
    val_feed_name VARCHAR(100),
    val_feed_url VARCHAR(255),
    val_feed_auth_user VARCHAR(100),
    val_feed_auth_pass VARCHAR(255),
    val_feed_notes LONGVARCHAR,
    val_feed_archived BOOLEAN,
    val_feed_update_enabled BOOLEAN,
    val_feed_update_interval BIGINT,
    val_feed_downloaded_at TIMESTAMP WITH TIME ZONE,
    val_feed_published_at TIMESTAMP WITH TIME ZONE,
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_news_feeds
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Neuen Feed erzeugen
    IF val_feed_id < 1 THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Feed speichern
      INSERT INTO immotool_news_feeds (
        feed_name,
        feed_url,
        feed_auth_user,
        feed_auth_pass,
        feed_notes,
        feed_archived,
        feed_update_enabled,
        feed_update_interval,
        feed_downloaded_at,
        feed_published_at,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_feed_name,
        val_feed_url,
        val_feed_auth_user,
        val_feed_auth_pass,
        val_feed_notes,
        val_feed_archived,
        val_feed_update_enabled,
        val_feed_update_interval,
        val_feed_downloaded_at,
        val_feed_published_at,
        owner_uid,
        owner_gid,
        permissions);

      -- ID zurückliefern
      SET val_feed_id = IDENTITY();

    -- Bestehenden Feed bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_news_feeds
        WHERE feed_id = val_feed_id;

      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>1 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>1 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>1 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Feed bearbeiten
      UPDATE immotool_news_feeds
        SET
          feed_name = val_feed_name,
          feed_url = val_feed_url,
          feed_auth_user = val_feed_auth_user,
          feed_auth_pass = val_feed_auth_pass,
          feed_notes = val_feed_notes,
          feed_archived = val_feed_archived,
          feed_update_enabled = val_feed_update_enabled,
          feed_update_interval = val_feed_update_interval,
          feed_downloaded_at = val_feed_downloaded_at,
          feed_published_at = val_feed_published_at,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions,
          modified_at = NOW()
        WHERE
          feed_id = val_feed_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_news_feeds
  TO "IMMOTOOL";
