-- -----------------------------------------------------
-- Contacts-Addon for OpenEstate-ImmoTool
-- routines for HSQLDB
-- Copyright (C) 2009-2019 OpenEstate.org
-- -----------------------------------------------------
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_contacts_groups
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_contacts_groups( val_group_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_contacts_groups
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_contacts_groups
      WHERE group_id = val_group_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_read_immotool_contacts_groups
-- -----------------------------------------------------
CREATE FUNCTION can_read_immotool_contacts_groups( val_group_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_read_immotool_contacts_groups
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_contacts_groups
      WHERE group_id = val_group_id;

    RETURN can_read( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_contacts_groups
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_contacts_groups( val_group_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_contacts_groups
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_contacts_groups
      WHERE group_id = val_group_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_contacts_entries
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_contacts_entries( val_entry_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_contacts_entries
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE entry_group_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Adress-Gruppe ermitteln
    SELECT group_id
      INTO entry_group_id
      FROM immotool_contacts_entries
      WHERE entry_id = val_entry_id;

    -- Zum Löschen eines Adress-Eintrages müssen Schreibrechte auf der Adress-Gruppe vorliegen
    RETURN can_write_immotool_contacts_groups( entry_group_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_read_immotool_contacts_entries
-- -----------------------------------------------------
CREATE FUNCTION can_read_immotool_contacts_entries( val_entry_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_read_immotool_contacts_entries
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE entry_group_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Adress-Gruppe ermitteln
    SELECT group_id
      INTO entry_group_id
      FROM immotool_contacts_entries
      WHERE entry_id = val_entry_id;

    -- Zum Lesen eines Adress-Eintrages müssen Lesenrechte auf der Adress-Gruppe vorliegen
    RETURN can_read_immotool_contacts_groups( entry_group_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_contacts_entries
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_contacts_entries( val_entry_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_contacts_entries
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE entry_group_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Adress-Gruppe ermitteln
    SELECT group_id
      INTO entry_group_id
      FROM immotool_contacts_entries
      WHERE entry_id = val_entry_id;

    -- Zum Bearbeiten eines Adress-Eintrages müssen Schreibrechte auf der Adress-Gruppe vorliegen
    RETURN can_write_immotool_contacts_groups( entry_group_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function get_immotool_contacts_entry_id
-- -----------------------------------------------------
CREATE FUNCTION get_immotool_contacts_entry_id( val_public_id VARCHAR(255) )
  RETURNS BIGINT
  SPECIFIC get_immotool_contacts_entry_id
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE entry_id BIGINT;

    -- Adress-ID aus Adress-Nr ermitteln
    SELECT entry_id
      INTO entry_id
      FROM immotool_contacts_entries
      WHERE entry_public_id IS NOT NULL AND entry_public_id = val_public_id;

    -- ID der ermittelten Adresse zurückliefern
    RETURN entry_id;
  END;
.;

GRANT EXECUTE
  ON FUNCTION get_immotool_contacts_entry_id
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_contacts_entries
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_contacts_entries( val_entry_id BIGINT )
  SPECIFIC remove_immotool_contacts_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_contacts_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag entfernen
    DELETE
      FROM immotool_contacts_entries
      WHERE entry_id = val_entry_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_contacts_entries
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_contacts_entries_address
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_contacts_entries_address(
    val_entry_id BIGINT,
    val_setting_key VARCHAR(255)
  )
  SPECIFIC remove_immotool_contacts_entries_address
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_contacts_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Alle Settings des Eintrages entfernen
    IF val_setting_key IS NULL THEN
      DELETE
        FROM immotool_contacts_entries_settings
        WHERE entry_id = val_entry_id
        AND setting_key LIKE 'entry.%';

    -- Spezifisches Setting des Eintrages entfernen
    ELSE
      DELETE
        FROM immotool_contacts_entries_settings
        WHERE entry_id = val_entry_id
        AND setting_key LIKE 'entry.%'
        AND setting_key = val_setting_key;
    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_contacts_entries_address
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_contacts_entries_settings
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_contacts_entries_settings(
    val_entry_id BIGINT,
    val_setting_key VARCHAR(255)
  )
  SPECIFIC remove_immotool_contacts_entries_settings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_contacts_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Alle Settings des Eintrages entfernen
    IF val_setting_key IS NULL THEN
      DELETE
        FROM immotool_contacts_entries_settings
        WHERE entry_id = val_entry_id
        AND setting_key NOT LIKE 'entry.%';

    -- Spezifisches Setting des Eintrages entfernen
    ELSE
      DELETE
        FROM immotool_contacts_entries_settings
        WHERE entry_id = val_entry_id
        AND setting_key NOT LIKE 'entry.%'
        AND setting_key = val_setting_key;
    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_contacts_entries_settings
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_contacts_groups
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_contacts_groups( val_group_id BIGINT )
  SPECIFIC remove_immotool_contacts_groups
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_group_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_contacts_groups( val_group_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Gruppe entfernen
    DELETE
      FROM immotool_contacts_groups
      WHERE group_id = val_group_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_contacts_groups
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_contacts_entries
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_contacts_entries(
    INOUT val_entry_id BIGINT,
    val_group_id BIGINT,
    val_entry_public_id VARCHAR(255),
    val_entry_notes LONGVARCHAR,
    val_entry_active BOOLEAN,
    val_entry_birthday DATE
  )
  SPECIFIC save_immotool_contacts_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Prüfen ob Schreib-Rechte auf der Adress-Gruppe vorliegen
    IF val_group_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;
    SET allowed = can_write_immotool_contacts_groups( val_group_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuen Eintrag erzeugen
    IF val_entry_id < 1 THEN

      -- Eintrag speichern
      INSERT INTO immotool_contacts_entries (
        group_id,
        entry_public_id,
        entry_notes,
        entry_active,
        entry_birthday )
      VALUES (
        val_group_id,
        val_entry_public_id,
        val_entry_notes,
        val_entry_active,
        val_entry_birthday );

      -- ID zurückliefern
      SET val_entry_id = IDENTITY();

    -- Bestehenden Eintrag bearbeiten
    ELSE

      -- Eintrag bearbeiten
      UPDATE immotool_contacts_entries
        SET
          group_id = val_group_id,
          entry_public_id = val_entry_public_id,
          entry_notes = val_entry_notes,
          entry_active = val_entry_active,
          entry_birthday = val_entry_birthday,
          modified_at = NOW()
        WHERE
          entry_id = val_entry_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_contacts_entries
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_contacts_entries_logs
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_contacts_entries_logs(
    val_log_id BIGINT,
    val_entry_id BIGINT
  )
  SPECIFIC save_immotool_contacts_entries_logs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_contacts_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Protokolleintrag aktualisieren
    DELETE
      FROM immotool_contacts_entries_logs
      WHERE log_id = val_log_id;

    INSERT
      INTO immotool_contacts_entries_logs (log_id, entry_id)
      VALUES (val_log_id, val_entry_id);

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_contacts_entries_logs
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_contacts_entries_settings
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_contacts_entries_settings(
    val_entry_id BIGINT,
    val_setting_key VARCHAR(255),
    val_setting_value LONGVARCHAR
  )
  SPECIFIC save_immotool_contacts_entries_settings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_contacts_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Existierenden Eintrag entfernen, wenn vorhanden
    DELETE FROM immotool_contacts_entries_settings
      WHERE entry_id = val_entry_id
      AND setting_key = val_setting_key;

    -- Eintrag speichern, wenn ein Wert angegeben wurde
    IF (val_setting_value IS NOT NULL) THEN

      INSERT INTO immotool_contacts_entries_settings (
        entry_id,
        setting_key,
        setting_value )
      VALUES (
        val_entry_id,
        val_setting_key,
        val_setting_value );

  	END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_contacts_entries_settings
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_contacts_groups
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_contacts_groups(
    INOUT val_group_id BIGINT,
    val_group_name VARCHAR(100),
    val_group_notes LONGVARCHAR,
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_contacts_groups
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Neue Gruppe erzeugen
    IF val_group_id < 1 THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Gruppe speichern
      INSERT INTO immotool_contacts_groups (
        group_name,
        group_notes,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_group_name,
        val_group_notes,
        owner_uid,
        owner_gid,
        permissions );

      -- ID zurückliefern
      SET val_group_id = IDENTITY();

    -- Bestehende Gruppe bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_contacts_groups
        WHERE group_id = val_group_id;

      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>0 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN
        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>0 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>=0 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Eintrag bearbeiten
      UPDATE immotool_contacts_groups
        SET
          group_name = val_group_name,
          group_notes = val_group_notes,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions,
          modified_at = NOW()
        WHERE
          group_id = val_group_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_contacts_groups
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure set_immotool_contacts_entries_active
-- -----------------------------------------------------
CREATE PROCEDURE set_immotool_contacts_entries_active(
    val_entry_id BIGINT,
    val_entry_active BOOLEAN
  )
  SPECIFIC set_immotool_contacts_entries_active
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_contacts_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag bearbeiten
    UPDATE immotool_contacts_entries
      SET
        entry_active = val_entry_active,
        modified_at = NOW()
      WHERE
        entry_id = val_entry_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE set_immotool_contacts_entries_active
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure set_immotool_contacts_entries_group
-- -----------------------------------------------------
CREATE PROCEDURE set_immotool_contacts_entries_group(
    val_entry_id BIGINT,
    val_group_id BIGINT
  )
  SPECIFIC set_immotool_contacts_entries_group
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte auf der Adresse besitzt
    SET allowed = can_write_immotool_contacts_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte auf der neuen Adress-Gruppe besitzt
    SET allowed = can_write_immotool_contacts_groups( val_group_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag bearbeiten
    UPDATE immotool_contacts_entries
      SET
        group_id = val_group_id,
        modified_at = NOW()
      WHERE
        entry_id = val_entry_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE set_immotool_contacts_entries_group
  TO "IMMOTOOL";
