-- -----------------------------------------------------
-- Mail-Addon for OpenEstate-ImmoTool
-- routines for HSQLDB
-- Copyright (C) 2009-2019 OpenEstate.org
-- -----------------------------------------------------
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_mails_accounts
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_mails_accounts( val_account_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_mails_accounts
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_mails_accounts
      WHERE account_id = val_account_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_read_immotool_mails_accounts
-- -----------------------------------------------------
CREATE FUNCTION can_read_immotool_mails_accounts( val_account_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_read_immotool_mails_accounts
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_mails_accounts
      WHERE account_id = val_account_id;

    RETURN can_read( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_mails_accounts
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_mails_accounts( val_account_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_mails_accounts
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_mails_accounts
      WHERE account_id = val_account_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_mails_folders
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_mails_folders( val_folder_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_mails_folders
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_account_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Konto ermitteln
    SELECT account_id
      INTO val_account_id
      FROM immotool_mails_folders
      WHERE folder_id = val_folder_id;

    -- Zum Löschen eines Mail-Ordners müssen Schreibrechte auf dem Mail-Konto vorliegen
    RETURN can_write_immotool_mails_accounts( val_account_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_read_immotool_mails_folders
-- -----------------------------------------------------
CREATE FUNCTION can_read_immotool_mails_folders( val_folder_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_read_immotool_mails_folders
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_account_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Konto ermitteln
    SELECT account_id
      INTO val_account_id
      FROM immotool_mails_folders
      WHERE folder_id = val_folder_id;

    -- Zum Lesen eines Mail-Ordners müssen Leserechte auf dem Mail-Konto vorliegen
    RETURN can_read_immotool_mails_accounts( val_account_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_mails_folders
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_mails_folders( val_folder_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_mails_folders
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_account_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Konto ermitteln
    SELECT account_id
      INTO val_account_id
      FROM immotool_mails_folders
      WHERE folder_id = val_folder_id;

    -- Zum Schreiben eines Mail-Ordners müssen Schreibrechte auf dem Mail-Konto vorliegen
    RETURN can_write_immotool_mails_accounts( val_account_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_mails_messages
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_mails_messages( val_message_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_mails_messages
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_folder_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Ordner ermitteln
    SELECT folder_id
      INTO val_folder_id
      FROM immotool_mails_messages
      WHERE message_id = val_message_id;

    -- Zum Löschen einer Mail-Mitteilung müssen Schreibrechte auf dem Mail-Ordner vorliegen
    RETURN can_write_immotool_mails_folders( val_folder_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_read_immotool_mails_messages
-- -----------------------------------------------------
CREATE FUNCTION can_read_immotool_mails_messages( val_message_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_read_immotool_mails_messages
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_folder_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Ordner ermitteln
    SELECT folder_id
      INTO val_folder_id
      FROM immotool_mails_messages
      WHERE message_id = val_message_id;

    -- Zum Lesen einer Mail-Mitteilung müssen Leserechte auf dem Mail-Ordner vorliegen
    RETURN can_read_immotool_mails_folders( val_folder_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_mails_messages
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_mails_messages( val_message_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_mails_messages
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_folder_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Ordner ermitteln
    SELECT folder_id
      INTO val_folder_id
      FROM immotool_mails_messages
      WHERE message_id = val_message_id;

    -- Zum Löschen einer Mail-Mitteilung müssen Schreibrechte auf dem Mail-Ordner vorliegen
    RETURN can_write_immotool_mails_folders( val_folder_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_mails_parts
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_mails_parts( val_part_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_mails_parts
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_message_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Mitteilung ermitteln
    SELECT message_id
      INTO val_message_id
      FROM immotool_mails_parts
      WHERE part_id = val_part_id;

    -- Zum Löschen eines Mail-Parts müssen Schreibrechte auf der Mail-Mitteilung vorliegen
    RETURN can_write_immotool_mails_messages( val_message_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_read_immotool_mails_parts
-- -----------------------------------------------------
CREATE FUNCTION can_read_immotool_mails_parts( val_part_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_read_immotool_mails_parts
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_message_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Mitteilung ermitteln
    SELECT message_id
      INTO val_message_id
      FROM immotool_mails_parts
      WHERE part_id = val_part_id;

    -- Zum Lesen eines Mail-Parts müssen Leserechte auf der Mail-Mitteilung vorliegen
    RETURN can_read_immotool_mails_messages( val_message_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_mails_parts
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_mails_parts( val_part_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_mails_parts
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE val_message_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Mail-Mitteilung ermitteln
    SELECT message_id
      INTO val_message_id
      FROM immotool_mails_parts
      WHERE part_id = val_part_id;

    -- Zum Löschen eines Mail-Parts müssen Schreibrechte auf der Mail-Mitteilung vorliegen
    RETURN can_write_immotool_mails_messages( val_message_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_immotool_mails_accounts
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_mails_accounts( val_account_id BIGINT )
  SPECIFIC remove_immotool_mails_accounts
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_account_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_mails_accounts( val_account_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Mailkonto entfernen
    DELETE
      FROM immotool_mails_accounts
      WHERE account_id = val_account_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_mails_accounts
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_mails_accounts_settings
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_mails_accounts_settings(
    val_account_id BIGINT,
    val_setting_key VARCHAR(255)
  )
  SPECIFIC remove_immotool_mails_accounts_settings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_account_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_mails_accounts( val_account_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Alle Settings des Mailkontos entfernen
	  IF val_setting_key IS NULL THEN

	    DELETE
        FROM immotool_mails_accounts_settings
	      WHERE account_id = val_account_id;

		-- Spezifisches Setting des Mailkontos entfernen
	  ELSE

	    DELETE
        FROM immotool_mails_accounts_settings
	      WHERE account_id = val_account_id
	      AND setting_key = val_setting_key;

  	END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_mails_accounts_settings
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_mails_addresses
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_mails_addresses( val_message_id BIGINT )
  SPECIFIC remove_immotool_mails_addresses
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Adressen entfernen
    DELETE
      FROM immotool_mails_addresses
      WHERE message_id = val_message_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_mails_addresses
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_mails_folders
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_mails_folders( val_folder_id BIGINT )
  SPECIFIC remove_immotool_mails_folders
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_folder_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_delete_immotool_mails_folders( val_folder_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Ordner entfernen
    DELETE
      FROM immotool_mails_folders
      WHERE folder_id = val_folder_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_mails_folders
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_mails_headers
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_mails_headers( val_part_id BIGINT )
  SPECIFIC remove_immotool_mails_headers
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_part_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_parts( val_part_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Header entfernen
    DELETE
      FROM immotool_mails_headers
      WHERE part_id = val_part_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_mails_headers
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_mails_messages
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_mails_messages( val_message_id BIGINT )
  SPECIFIC remove_immotool_mails_messages
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_delete_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Mail entfernen
    DELETE
      FROM immotool_mails_messages
      WHERE message_id = val_message_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_mails_messages
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_mails_parts
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_mails_parts( val_part_id BIGINT )
  SPECIFIC remove_immotool_mails_parts
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_part_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_delete_immotool_mails_parts( val_part_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Part entfernen
    DELETE
      FROM immotool_mails_parts
      WHERE part_id = val_part_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_mails_parts
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_mails_accounts
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_mails_accounts(
    INOUT val_account_id BIGINT,
    val_account_name VARCHAR(100),
    val_account_type VARCHAR(50),
    val_account_notes LONGVARCHAR,
    val_account_enabled BOOLEAN,
    val_account_auto_retrieval BOOLEAN,
    val_account_auto_retrieval_interval BIGINT,
    val_account_auto_retrieval_at TIMESTAMP WITH TIME ZONE,
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_mails_accounts
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Neues Mailkonto erzeugen
    IF val_account_id < 1 THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Mailkonto speichern
      INSERT INTO immotool_mails_accounts (
        account_name,
        account_type,
        account_notes,
        account_enabled,
        account_auto_retrieval,
        account_auto_retrieval_interval,
        account_auto_retrieval_at,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_account_name,
        val_account_type,
        val_account_notes,
        val_account_enabled,
        val_account_auto_retrieval,
        val_account_auto_retrieval_interval,
        val_account_auto_retrieval_at,
        owner_uid,
        owner_gid,
        permissions );

      -- ID zurückliefern
      SET val_account_id = IDENTITY();

    -- Bestehendes Mailkonto bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_mails_accounts
        WHERE account_id = val_account_id;

      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>0 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>0 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>=0 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Mailkonto bearbeiten
      UPDATE immotool_mails_accounts
        SET
          account_name = val_account_name,
          account_type = val_account_type,
          account_notes = val_account_notes,
          account_enabled = val_account_enabled,
          account_auto_retrieval = val_account_auto_retrieval,
          account_auto_retrieval_interval = val_account_auto_retrieval_interval,
          account_auto_retrieval_at = val_account_auto_retrieval_at,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions,
          modified_at = NOW()
        WHERE
          account_id = val_account_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_mails_accounts
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_mails_accounts_settings
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_mails_accounts_settings(
    val_account_id BIGINT,
    val_setting_key VARCHAR(255),
    val_setting_value LONGVARCHAR
  )
  SPECIFIC save_immotool_mails_accounts_settings
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_account_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_accounts( val_account_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Existierenden Eintrag entfernen, wenn vorhanden
    DELETE
      FROM immotool_mails_accounts_settings
      WHERE account_id = val_account_id
      AND setting_key = val_setting_key;

    -- Eintrag speichern, wenn ein Wert angegeben wurde
    IF (val_setting_value IS NOT NULL) THEN

      INSERT INTO immotool_mails_accounts_settings (
        account_id,
        setting_key,
        setting_value )
      VALUES (
        val_account_id,
        val_setting_key,
        val_setting_value );

  	END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_mails_accounts_settings
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_mails_addresses
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_mails_addresses(
    val_message_id BIGINT,
    val_address_type VARCHAR(8),
    val_address_value VARCHAR(255),
    val_address_person VARCHAR(255)
  )
  SPECIFIC save_immotool_mails_addresses
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Adresse speichern
    INSERT INTO immotool_mails_addresses (
      message_id,
      address_type,
      address_value,
      address_person )
    VALUES (
      val_message_id,
      val_address_type,
      val_address_value,
      val_address_person );

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_mails_addresses
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_mails_folders
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_mails_folders(
    INOUT val_folder_id BIGINT,
    val_account_id BIGINT,
    val_folder_parent_id BIGINT,
    val_folder_name VARCHAR(255),
    val_folder_holds_messages BOOLEAN,
    val_folder_holds_folders BOOLEAN,
    val_folder_auto_retrieval BOOLEAN,
    val_folder_special_type VARCHAR(50)
  )
  SPECIFIC save_immotool_mails_folders
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_account_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_accounts( val_account_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuen Ordner erzeugen
    IF val_folder_id < 1 THEN
      -- Ordner speichern
      INSERT INTO immotool_mails_folders (
        account_id,
        folder_parent_id,
        folder_name,
        folder_holds_messages,
        folder_holds_folders,
        folder_auto_retrieval,
        folder_special_type )
      VALUES (
        val_account_id,
        val_folder_parent_id,
        val_folder_name,
        val_folder_holds_messages,
        val_folder_holds_folders,
        val_folder_auto_retrieval,
        val_folder_special_type );

      -- ID zurückliefern
      SET val_folder_id = IDENTITY();

    -- Bestehenden Ordner bearbeiten
    ELSE
      -- Ordner bearbeiten
      UPDATE immotool_mails_folders
        SET
          account_id = val_account_id,
          folder_parent_id = val_folder_parent_id,
          folder_name = val_folder_name,
          folder_holds_messages = val_folder_holds_messages,
          folder_holds_folders = val_folder_holds_folders,
          folder_auto_retrieval = val_folder_auto_retrieval,
          folder_special_type = val_folder_special_type,
          modified_at = NOW()
        WHERE
          folder_id = val_folder_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_mails_folders
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_mails_headers
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_mails_headers(
    val_part_id BIGINT,
    val_header_key VARCHAR(255),
    val_header_value LONGVARCHAR
  )
  SPECIFIC save_immotool_mails_headers
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_part_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_parts( val_part_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Header speichern
    INSERT INTO immotool_mails_headers (
      part_id,
      header_key,
      header_value )
    VALUES (
      val_part_id,
      val_header_key,
      val_header_value );

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_mails_headers
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_mails_messages
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_mails_messages(
    INOUT val_message_id BIGINT,
    val_folder_id BIGINT,
    val_message_subject VARCHAR(255),
    val_message_from VARCHAR(255),
    val_message_to VARCHAR(255),
    val_message_size BIGINT,
    val_message_uid BIGINT,
    val_message_uid_validity BIGINT,
    val_message_mime_id VARCHAR(255),
    val_message_hash VARCHAR(255),
    val_message_in_reply_to VARCHAR(255),
    val_message_is_answered BOOLEAN,
    val_message_is_archived BOOLEAN,
    val_message_is_downloaded BOOLEAN,
    val_message_is_draft BOOLEAN,
    val_message_is_flagged BOOLEAN,
    val_message_is_forwarded BOOLEAN,
    val_message_is_seen BOOLEAN,
    val_message_is_received BOOLEAN,
    val_message_has_attachments BOOLEAN,
    val_message_mdn_status VARCHAR(25),
    val_message_sent_at TIMESTAMP WITH TIME ZONE,
    val_message_received_at TIMESTAMP WITH TIME ZONE
  )
  SPECIFIC save_immotool_mails_messages
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_folder_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_folders( val_folder_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neue Mail erzeugen
    IF val_message_id < 1 THEN
      -- Mail speichern
      INSERT INTO immotool_mails_messages (
        folder_id,
        message_subject,
        message_from,
        message_to,
        message_size,
        message_uid,
        message_uid_validity,
        message_mime_id,
        message_hash,
        message_in_reply_to,
        message_is_answered,
        message_is_archived,
        message_is_downloaded,
        message_is_draft,
        message_is_flagged,
        message_is_forwarded,
        message_is_seen,
        message_is_received,
        message_has_attachments,
        message_mdn_status,
        message_sent_at,
        message_received_at )
      VALUES (
        val_folder_id,
        val_message_subject,
        val_message_from,
        val_message_to,
        val_message_size,
        val_message_uid,
        val_message_uid_validity,
        val_message_mime_id,
        val_message_hash,
        val_message_in_reply_to,
        val_message_is_answered,
        val_message_is_archived,
        val_message_is_downloaded,
        val_message_is_draft,
        val_message_is_flagged,
        val_message_is_forwarded,
        val_message_is_seen,
        val_message_is_received,
        val_message_has_attachments,
        val_message_mdn_status,
        val_message_sent_at,
        val_message_received_at );

      -- ID zurückliefern
      SET val_message_id = IDENTITY();

    -- Bestehende Mail bearbeiten
    ELSE
      -- Mail bearbeiten
      UPDATE immotool_mails_messages
        SET
          folder_id = val_folder_id,
          message_subject = val_message_subject,
          message_from = val_message_from,
          message_to = val_message_to,
          message_size = val_message_size,
          message_uid = val_message_uid,
          message_uid_validity = val_message_uid_validity,
          message_mime_id = val_message_mime_id,
          message_hash = val_message_hash,
          message_in_reply_to = val_message_in_reply_to,
          message_is_answered = val_message_is_answered,
          message_is_archived = val_message_is_archived,
          message_is_downloaded = val_message_is_downloaded,
          message_is_draft = val_message_is_draft,
          message_is_flagged = val_message_is_flagged,
          message_is_forwarded = val_message_is_forwarded,
          message_is_seen = val_message_is_seen,
          message_is_received = val_message_is_received,
          message_has_attachments = val_message_has_attachments,
          message_mdn_status = val_message_mdn_status,
          message_sent_at = val_message_sent_at,
          message_received_at = val_message_received_at,
          modified_at = NOW()
        WHERE
          message_id = val_message_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_mails_messages
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_mails_parts
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_mails_parts(
    INOUT val_part_id BIGINT,
    val_message_id BIGINT,
    val_part_parent_id BIGINT,
    val_part_type VARCHAR(255),
    val_part_description VARCHAR(255),
    val_part_disposition VARCHAR(255),
    val_part_text LONGVARCHAR,
    val_part_file_name VARCHAR(255),
    val_part_file_size BIGINT,
    val_part_file_content BLOB(1G)
  )
  SPECIFIC save_immotool_mails_parts
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuen Part erzeugen
    IF val_part_id < 1 THEN
      -- Part speichern
      INSERT INTO immotool_mails_parts (
        message_id,
        part_parent_id,
        part_type,
        part_description,
        part_disposition,
        part_text,
        part_file_name,
        part_file_size )
      VALUES (
        val_message_id,
        val_part_parent_id,
        val_part_type,
        val_part_description,
        val_part_disposition,
        val_part_text,
        val_part_file_name,
        val_part_file_size );

      -- ID zurückliefern
      SET val_part_id = IDENTITY();

    -- Bestehenden Part bearbeiten
    ELSE
      -- Part bearbeiten
      UPDATE immotool_mails_parts
        SET
          message_id = val_message_id,
          part_parent_id = val_part_parent_id,
          part_type = val_part_type,
          part_description = val_part_description,
          part_disposition = val_part_disposition,
          part_text = val_part_text,
          part_file_name = val_part_file_name,
          part_file_size = val_part_file_size,
          modified_at = NOW()
        WHERE
          part_id = val_part_id;

    END IF;

    -- Datei ggf. aktualisieren
    IF val_part_file_content IS NOT NULL THEN

      UPDATE immotool_mails_parts
        SET
          part_file_content = val_part_file_content,
          modified_at = NOW()
        WHERE
          part_id = val_part_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_mails_parts
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure set_immotool_mails_messages_answered
-- -----------------------------------------------------
CREATE PROCEDURE set_immotool_mails_messages_answered(
    val_message_id BIGINT,
    val_message_is_answered BOOLEAN
  )
  SPECIFIC set_immotool_mails_messages_answered
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Mail bearbeiten
    UPDATE immotool_mails_messages
      SET
        message_is_answered = val_message_is_answered,
        modified_at = NOW()
      WHERE
        message_id = val_message_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE set_immotool_mails_messages_answered
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure set_immotool_mails_messages_archived
-- -----------------------------------------------------
CREATE PROCEDURE set_immotool_mails_messages_archived(
    val_message_id BIGINT,
    val_message_is_archived BOOLEAN
  )
  SPECIFIC set_immotool_mails_messages_archived
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Mail bearbeiten
    UPDATE immotool_mails_messages
      SET
        message_is_archived = val_message_is_archived,
        modified_at = NOW()
      WHERE
        message_id = val_message_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE set_immotool_mails_messages_archived
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure set_immotool_mails_messages_forwarded
-- -----------------------------------------------------
CREATE PROCEDURE set_immotool_mails_messages_forwarded(
    val_message_id BIGINT,
    val_message_is_forwarded BOOLEAN
  )
  SPECIFIC set_immotool_mails_messages_forwarded
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Mail bearbeiten
    UPDATE immotool_mails_messages
      SET
        message_is_forwarded = val_message_is_forwarded,
        modified_at = NOW()
      WHERE
        message_id = val_message_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE set_immotool_mails_messages_forwarded
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure set_immotool_mails_messages_mdn
-- -----------------------------------------------------
CREATE PROCEDURE set_immotool_mails_messages_mdn(
    val_message_id BIGINT,
    val_message_mdn_status VARCHAR(25)
  )
  SPECIFIC set_immotool_mails_messages_mdn
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Mail bearbeiten
    UPDATE immotool_mails_messages
      SET
        message_mdn_status = val_message_mdn_status,
        modified_at = NOW()
      WHERE
        message_id = val_message_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE set_immotool_mails_messages_mdn
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure set_immotool_mails_messages_seen
-- -----------------------------------------------------
CREATE PROCEDURE set_immotool_mails_messages_seen(
    val_message_id BIGINT,
    val_message_is_seen BOOLEAN
  )
  SPECIFIC set_immotool_mails_messages_seen
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_message_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF NOT can_write_immotool_mails_messages( val_message_id ) = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Mail bearbeiten
    UPDATE immotool_mails_messages
      SET
        message_is_seen = val_message_is_seen,
        modified_at = NOW()
      WHERE
        message_id = val_message_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE set_immotool_mails_messages_seen
  TO "IMMOTOOL";
