-- -----------------------------------------------------
-- Calendar-Addon for OpenEstate-ImmoTool
-- routines for HSQLDB
-- Copyright (C) 2009-2019 OpenEstate.org
-- -----------------------------------------------------
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_calendar_groups
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_calendar_groups( val_group_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_calendar_groups
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_calendar_groups
      WHERE group_id = val_group_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_read_immotool_calendar_groups
-- -----------------------------------------------------
CREATE FUNCTION can_read_immotool_calendar_groups( val_group_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_read_immotool_calendar_groups
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_calendar_groups
      WHERE group_id = val_group_id;

    RETURN can_read( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_calendar_groups
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_calendar_groups( val_group_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_calendar_groups
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_calendar_groups
      WHERE group_id = val_group_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_calendar_entries
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_calendar_entries( val_entry_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_calendar_entries
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE entry_group_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Kalender-Gruppe ermitteln
    SELECT group_id
      INTO entry_group_id
      FROM immotool_calendar_entries
      WHERE entry_id = val_entry_id;

    -- Zum Löschen eines Kalender-Eintrages müssen Schreibrechte auf der Kalender-Gruppe vorliegen
    RETURN can_write_immotool_calendar_groups( entry_group_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_read_immotool_calendar_entries
-- -----------------------------------------------------
CREATE FUNCTION can_read_immotool_calendar_entries( val_entry_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_read_immotool_calendar_entries
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE entry_group_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Kalender-Gruppe ermitteln
    SELECT group_id
      INTO entry_group_id
      FROM immotool_calendar_entries
      WHERE entry_id = val_entry_id;

    -- Zum Lesen eines Kalender-Eintrages müssen Leserechte auf der Kalender-Gruppe vorliegen
    RETURN can_read_immotool_calendar_groups( entry_group_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_calendar_entries
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_calendar_entries( val_entry_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_calendar_entries
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE entry_group_id BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- Kalender-Gruppe ermitteln
    SELECT group_id
      INTO entry_group_id
      FROM immotool_calendar_entries
      WHERE entry_id = val_entry_id;

    -- Zum Schreiben eines Kalender-Eintrages müssen Schreibrechte auf der Kalender-Gruppe vorliegen
    RETURN can_write_immotool_calendar_groups( entry_group_id );
  END;
.;
\.
-- -----------------------------------------------------
-- Procedure remove_immotool_calendar_entries
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_calendar_entries( val_entry_id BIGINT )
  SPECIFIC remove_immotool_calendar_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_calendar_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag entfernen
    DELETE
      FROM immotool_calendar_entries
      WHERE entry_id = val_entry_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_calendar_entries
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_calendar_groups
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_calendar_groups( val_group_id BIGINT )
  SPECIFIC remove_immotool_calendar_groups
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_group_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_calendar_groups( val_group_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Gruppe entfernen
    DELETE
      FROM immotool_calendar_groups
      WHERE group_id = val_group_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_calendar_groups
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_calendar_entries
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_calendar_entries(
    INOUT val_entry_id BIGINT,
    val_group_id BIGINT,
    val_entry_uid VARCHAR(255),
    val_entry_type VARCHAR(45),
    val_entry_status VARCHAR(45),
    val_entry_title VARCHAR(255),
    val_entry_begin TIMESTAMP WITH TIME ZONE,
    val_entry_end TIMESTAMP WITH TIME ZONE,
    val_entry_notes LONGVARCHAR,
    val_entry_ical LONGVARCHAR
  )
  SPECIFIC save_immotool_calendar_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Prüfen ob Schreib-Rechte auf der Kalender-Gruppe vorliegen
    IF val_group_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;
    SET allowed = can_write_immotool_calendar_groups( val_group_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuen Eintrag erzeugen
    IF val_entry_id < 1 THEN

      -- Eintrag speichern
      INSERT INTO immotool_calendar_entries (
        group_id,
        entry_uid,
        entry_type,
        entry_status,
        entry_title,
        entry_begin,
        entry_end,
        entry_notes,
        entry_ical )
      VALUES (
        val_group_id,
        val_entry_uid,
        val_entry_type,
        val_entry_status,
        val_entry_title,
        val_entry_begin,
        val_entry_end,
        val_entry_notes,
        val_entry_ical );

      -- ID zurückliefern
      SET val_entry_id = IDENTITY();

    -- Bestehenden Eintrag bearbeiten
    ELSE

      -- Eintrag bearbeiten
      UPDATE immotool_calendar_entries
        SET
          group_id = val_group_id,
          entry_uid = val_entry_uid,
          entry_type = val_entry_type,
          entry_status = val_entry_status,
          entry_title = val_entry_title,
          entry_begin = val_entry_begin,
          entry_end = val_entry_end,
          entry_notes = val_entry_notes,
          entry_ical = val_entry_ical,
          modified_at = NOW()
        WHERE
          entry_id = val_entry_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_calendar_entries
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_calendar_entries_logs
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_calendar_entries_logs(
    val_log_id BIGINT,
    val_entry_id BIGINT
  )
  SPECIFIC save_immotool_calendar_entries_logs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_calendar_entries( val_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Protokolleintrag aktualisieren
    DELETE
      FROM immotool_calendar_entries_logs
      WHERE log_id = val_log_id;

    INSERT
      INTO immotool_calendar_entries_logs (log_id, entry_id)
      VALUES (val_log_id, val_entry_id);

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_calendar_entries_logs
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_calendar_groups
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_calendar_groups(
    INOUT val_group_id BIGINT,
    val_group_name VARCHAR(100),
    val_group_notes LONGVARCHAR,
    val_group_url VARCHAR(255),
    val_group_type VARCHAR(45),
    val_group_auth_method VARCHAR(45),
    val_group_auth_user VARCHAR(100),
    val_group_auth_pass VARCHAR(255),
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_calendar_groups
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Neue Gruppe erzeugen
    IF val_group_id < 1 THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Gruppe speichern
      INSERT INTO immotool_calendar_groups (
        group_name,
        group_notes,
        group_url,
        group_type,
        group_auth_method,
        group_auth_user,
        group_auth_pass,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_group_name,
        val_group_notes,
        val_group_url,
        val_group_type,
        val_group_auth_method,
        val_group_auth_user,
        val_group_auth_pass,
        owner_uid,
        owner_gid,
        permissions );

      -- ID zurückliefern
      SET val_group_id = IDENTITY();

    -- Bestehende Gruppe bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_calendar_groups
        WHERE group_id = val_group_id;

      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>0 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>0 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>=0 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Gruppe bearbeiten
      UPDATE immotool_calendar_groups
        SET
          group_name = val_group_name,
          group_notes = val_group_notes,
          group_url = val_group_url,
          group_type = val_group_type,
          group_auth_method = val_group_auth_method,
          group_auth_user = val_group_auth_user,
          group_auth_pass = val_group_auth_pass,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions,
          modified_at = NOW()
        WHERE
          group_id = val_group_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_calendar_groups
  TO "IMMOTOOL";
