-- -----------------------------------------------------
-- Agency-Addon for OpenEstate-ImmoTool
-- update 5 for HSQLDB
-- Copyright (C) 2009-2019 OpenEstate.org
-- -----------------------------------------------------
-- Update field 'plugin_id' in 'immotool_logs'
-- Check field 'plugin_id' in 'view_immotool_agency_objects_logs'
-- Add a specific name to all procedures & functions
-- Bugfix within the can_* functions
-- -----------------------------------------------------

-- -----------------------------------------------------
-- Update field 'plugin_id' in 'immotool_logs'
-- -----------------------------------------------------
UPDATE immotool_logs
  SET plugin_id = (SELECT plugin_id FROM immotool_plugins WHERE plugin_name = 'OpenEstate-Tool-Agency')
  WHERE log_entry_type LIKE 'Agency.%';

-- -----------------------------------------------------
-- Check field 'plugin_id' in 'view_immotool_agency_objects_logs'
-- -----------------------------------------------------
ALTER VIEW view_immotool_agency_objects_logs AS
  SELECT a.object_id, b.* FROM immotool_agency_objects_logs a
  INNER JOIN immotool_logs b ON ( a.log_id = b.log_id )
  WHERE a.object_id IN (SELECT object_id FROM view_openestate_objects)
  AND b.plugin_id IN (SELECT plugin_id FROM view_immotool_plugins WHERE plugin_name='OpenEstate-Tool-Agency');

-- -----------------------------------------------------
-- Function can_delete_immotool_agency_contacts
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_delete_immotool_agency_contacts CASCADE;
\.
CREATE FUNCTION can_delete_immotool_agency_contacts( val_contact_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_contacts
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_contacts
      WHERE contact_id = val_contact_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_delete_immotool_agency_exports
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_delete_immotool_agency_exports CASCADE;
\.
CREATE FUNCTION can_delete_immotool_agency_exports( val_export_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_exports
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_exports
      WHERE export_id = val_export_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_delete_immotool_agency_exposes
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_delete_immotool_agency_exposes CASCADE;
\.
CREATE FUNCTION can_delete_immotool_agency_exposes( val_expose_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_exposes
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_exposes
      WHERE expose_id = val_expose_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_delete_immotool_agency_objects
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_delete_immotool_agency_objects CASCADE;
\.
CREATE FUNCTION can_delete_immotool_agency_objects( val_object_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_objects
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_objects
      WHERE object_id = val_object_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_delete_immotool_agency_searches
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_delete_immotool_agency_searches CASCADE;
\.
CREATE FUNCTION can_delete_immotool_agency_searches( val_search_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_searches
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_searches
      WHERE search_id = val_search_id;

    RETURN can_delete( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_write_immotool_agency_contacts
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_write_immotool_agency_contacts CASCADE;
\.
CREATE FUNCTION can_write_immotool_agency_contacts( val_contact_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_contacts
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_contacts
      WHERE contact_id = val_contact_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_write_immotool_agency_exports
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_write_immotool_agency_exports CASCADE;
\.
CREATE FUNCTION can_write_immotool_agency_exports( val_export_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_exports
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_exports
      WHERE export_id = val_export_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_write_immotool_agency_exposes
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_write_immotool_agency_exposes CASCADE;
\.
CREATE FUNCTION can_write_immotool_agency_exposes( val_expose_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_exposes
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_exposes
      WHERE expose_id = val_expose_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_write_immotool_agency_objects
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_write_immotool_agency_objects CASCADE;
\.
CREATE FUNCTION can_write_immotool_agency_objects( val_object_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_objects
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_objects
      WHERE object_id = val_object_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Function can_write_immotool_agency_searches
-- -----------------------------------------------------
DROP FUNCTION IF EXISTS can_write_immotool_agency_searches CASCADE;
\.
CREATE FUNCTION can_write_immotool_agency_searches( val_search_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_searches
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen
    SELECT access_permissions, access_owner_id, access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_searches
      WHERE search_id = val_search_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exports
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exports CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exports( val_export_id BIGINT )
  SPECIFIC remove_immotool_agency_exports
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_export_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_agency_exports( val_export_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Export entfernen
    DELETE
      FROM immotool_agency_exports
      WHERE export_id = val_export_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exports
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exports_entries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exports_entries CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exports_entries(
    val_export_id BIGINT,
    val_entry_object_id BIGINT
  )
  SPECIFIC remove_immotool_agency_exports_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_export_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_exports( val_export_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Spezifischen Eintrag entfernen
    IF val_entry_object_id IS NOT NULL THEN

      DELETE
        FROM immotool_agency_exports_entries
        WHERE export_id = val_export_id
        AND entry_object_id = val_entry_object_id;

    -- Alle Einträge entfernen
    ELSE

      DELETE
        FROM immotool_agency_exports_entries
        WHERE export_id = val_export_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exports_entries
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exports_logs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exports_logs CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exports_logs( val_log_id BIGINT )
  SPECIFIC remove_immotool_agency_exports_logs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_export_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_log_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT export_id
      INTO val_export_id
      FROM immotool_agency_exports_logs
      WHERE log_id = val_log_id;
    SET allowed = can_write_immotool_agency_exports( val_export_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Log entfernen
    DELETE
      FROM immotool_agency_exports_logs
      WHERE log_id = val_log_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exports_logs
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exports_logs_entries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exports_logs_entries CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exports_logs_entries(
    val_log_id BIGINT,
    val_entry_object_id BIGINT
  )
  SPECIFIC remove_immotool_agency_exports_logs_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_export_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_log_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT export_id
      INTO val_export_id
      FROM immotool_agency_exports_logs
      WHERE log_id = val_log_id;
    SET allowed = can_write_immotool_agency_exports( val_export_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Spezifischen Eintrag entfernen
    IF val_entry_object_id IS NOT NULL THEN

      DELETE
        FROM immotool_agency_exports_logs_entries
        WHERE log_id = val_log_id
        AND entry_object_id = val_entry_object_id;

    -- Einträge eines Logs entfernen
    ELSE

      DELETE
        FROM immotool_agency_exports_logs_entries
        WHERE log_id = val_log_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exports_logs_entries
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exports_publishings
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exports_publishings CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exports_publishings(
    val_export_id BIGINT,
    val_publishing_object_id BIGINT
  )
  SPECIFIC remove_immotool_agency_exports_publishings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_export_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_exports( val_export_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Spezifische Veröffentlichung entfernen
    IF val_publishing_object_id IS NOT NULL THEN

      DELETE
        FROM immotool_agency_exports_publishings
        WHERE export_id = val_export_id
        AND publishing_object_id = val_publishing_object_id;

    -- Veröffentlichungen eines Exportes entfernen
    ELSE

      DELETE
        FROM immotool_agency_exports_publishings
        WHERE export_id = val_export_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exports_publishings
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exports_settings
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exports_settings CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exports_settings(
    val_export_id BIGINT,
    val_setting_key VARCHAR(255)
  )
  SPECIFIC remove_immotool_agency_exports_settings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_export_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_exports( val_export_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Spezifisches Setting entfernen
    IF val_setting_key IS NOT NULL THEN

      DELETE
        FROM immotool_agency_exports_settings
        WHERE export_id = val_export_id
        AND setting_key = val_setting_key;

    -- Alle Settings entfernen
    ELSE

      DELETE
        FROM immotool_agency_exports_settings
        WHERE export_id = val_export_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exports_settings
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exposes
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exposes CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exposes( val_expose_id BIGINT )
  SPECIFIC remove_immotool_agency_exposes
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_expose_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_agency_exposes( val_expose_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Exposé entfernen
    DELETE
      FROM immotool_agency_exposes
      WHERE expose_id = val_expose_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exposes
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exposes_files
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exposes_files CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exposes_files( val_file_id BIGINT )
  SPECIFIC remove_immotool_agency_exposes_files
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_expose_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT expose_id
      INTO val_expose_id
      FROM immotool_agency_exposes_files
      WHERE file_id = val_file_id;
    SET allowed = can_write_immotool_agency_exposes( val_expose_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Datei entfernen
    DELETE
      FROM immotool_agency_exposes_files
      WHERE file_id = val_file_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exposes_files
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_exposes_settings
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_exposes_settings CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_exposes_settings(
    val_expose_id BIGINT,
    val_setting_key VARCHAR(2555)
  )
  SPECIFIC remove_immotool_agency_exposes_settings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_expose_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_exposes( val_expose_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- spezifisches Setting entfernen
    IF val_setting_key IS NOT NULL THEN

      DELETE
        FROM immotool_agency_exposes_settings
        WHERE expose_id = val_expose_id
        AND setting_key = val_setting_key;

    -- alle Settings entfernen
    ELSE

      DELETE
        FROM immotool_agency_exposes_settings
        WHERE expose_id = val_expose_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_exposes_settings
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_immotool_agency_objects_exports
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_immotool_agency_objects_exports CASCADE;
\.
CREATE PROCEDURE remove_immotool_agency_objects_exports(
    val_object_id BIGINT,
    val_export_id BIGINT
  )
  SPECIFIC remove_immotool_agency_objects_exports
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte auf einem der Objekte besitzt
    IF NOT allowed = TRUE AND val_object_id IS NOT NULL AND val_object_id > 0 THEN
      SET allowed = can_write_immotool_agency_objects( val_object_id );
    END IF;
    IF NOT allowed = TRUE AND val_export_id IS NOT NULL AND val_export_id > 0 THEN
      SET allowed = can_write_immotool_agency_exports( val_export_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- spezifische Verknüpfung entfernen
    IF val_object_id IS NOT NULL AND val_export_id IS NOT NULL THEN

      DELETE
        FROM immotool_agency_objects_exports
        WHERE object_id = val_object_id
        AND export_id = val_export_id;

    -- alle Verknüpfungen einer Immobilie entfernen
    ELSEIF val_object_id IS NOT NULL THEN

      DELETE
        FROM immotool_agency_objects_exports
        WHERE object_id = val_object_id;

    -- alle Verknüpfungen eines Exportes entfernen
    ELSEIF val_export_id IS NOT NULL THEN

      DELETE
        FROM immotool_agency_objects_exports
        WHERE export_id = val_export_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_objects_exports
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_contacts
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_contacts CASCADE;
\.
CREATE PROCEDURE remove_openestate_contacts( val_contact_id BIGINT )
  SPECIFIC remove_openestate_contacts
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_contact_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_agency_contacts( val_contact_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Ansprechpartner entfernen
    DELETE
      FROM openestate_contacts
      WHERE contact_id = val_contact_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_contacts
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_geodbs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_geodbs CASCADE;
\.
CREATE PROCEDURE remove_openestate_geodbs( val_geodb_id BIGINT )
  SPECIFIC remove_openestate_geodbs
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Geo-DB entfernen
    DELETE
      FROM openestate_geodbs
      WHERE geodb_id = val_geodb_id;

  END;
.;

-- -----------------------------------------------------
-- Procedure remove_openestate_geodbs_areas
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_geodbs_areas CASCADE;
\.
CREATE PROCEDURE remove_openestate_geodbs_areas( val_area_id BIGINT )
  SPECIFIC remove_openestate_geodbs_areas
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Region entfernen
    DELETE
      FROM openestate_geodbs_areas
      WHERE area_id = val_area_id;

  END;
.;

-- -----------------------------------------------------
-- Procedure remove_openestate_geodbs_countries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_geodbs_countries CASCADE;
\.
CREATE PROCEDURE remove_openestate_geodbs_countries( val_country_id BIGINT )
  SPECIFIC remove_openestate_geodbs_countries
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Land entfernen
    DELETE
      FROM openestate_geodbs_countries
      WHERE country_id = val_country_id;

  END;
.;

-- -----------------------------------------------------
-- Procedure remove_openestate_languages
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_languages CASCADE;
\.
CREATE PROCEDURE remove_openestate_languages( val_language_id BIGINT )
  SPECIFIC remove_openestate_languages
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Sprache entfernen
    DELETE
      FROM openestate_languages
      WHERE language_id = val_language_id;

  END;
.;

-- -----------------------------------------------------
-- Procedure remove_openestate_objects
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects CASCADE;
\.
CREATE PROCEDURE remove_openestate_objects( val_object_id BIGINT )
  SPECIFIC remove_openestate_objects
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Objekt entfernen
    DELETE
      FROM openestate_objects
      WHERE object_id = val_object_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_objects_attribs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects_attribs CASCADE;
\.
CREATE PROCEDURE remove_openestate_objects_attribs(
    val_object_id BIGINT,
    val_language_id BIGINT,
    val_attrib_group VARCHAR(255),
    val_attrib_name VARCHAR(255)
  )
  SPECIFIC remove_openestate_objects_attribs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Attribut in einer Sprache entfernen
    IF val_language_id IS NOT NULL AND val_attrib_group IS NOT NULL AND val_attrib_name IS NOT NULL THEN

      DELETE
        FROM openestate_objects_attribs
        WHERE object_id = val_object_id
        AND language_id = val_language_id
        AND attrib_group = val_attrib_group
        AND attrib_name = val_attrib_name;

    -- Attribut in allen Sprachen entfernen
    ELSEIF val_attrib_group IS NOT NULL AND val_attrib_name IS NOT NULL THEN

      DELETE
        FROM openestate_objects_attribs
        WHERE object_id = val_object_id
        AND attrib_group = val_attrib_group
        AND attrib_name = val_attrib_name;

    -- Attribute in allen Sprache entfernen
    ELSEIF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_attribs
        WHERE object_id = val_object_id
        AND language_id = val_language_id;

    -- Alle Attribute entfernen
    ELSE

      DELETE
        FROM openestate_objects_attribs
        WHERE object_id = val_object_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_attribs
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_objects_files
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects_files CASCADE;
\.
CREATE PROCEDURE remove_openestate_objects_files( val_file_id BIGINT )
  SPECIFIC remove_openestate_objects_files
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_files
      WHERE file_id = val_file_id;
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Datei entfernen
    DELETE
      FROM openestate_objects_files
      WHERE file_id = val_file_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_files
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_objects_files_i18n
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects_files_i18n CASCADE;
\.
CREATE PROCEDURE remove_openestate_objects_files_i18n(
    val_file_id BIGINT,
    val_language_id BIGINT
  )
  SPECIFIC remove_openestate_objects_files_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_files
      WHERE file_id = val_file_id;
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzungen zu einer Sprache entfernen
    IF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_files_i18n
        WHERE file_id = val_file_id
        AND language_id = val_language_id;

    -- Alle Übersetzungen entfernen
    ELSE

      DELETE
        FROM openestate_objects_files_i18n
        WHERE file_id = val_file_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_files_i18n
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_objects_geodbs_areas
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects_geodbs_areas CASCADE;
\.
CREATE PROCEDURE remove_openestate_objects_geodbs_areas(
    val_object_id BIGINT,
    val_area_id BIGINT
  )
  SPECIFIC remove_openestate_objects_geodbs_areas
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF val_area_id IS NOT NULL AND val_area_id > 0 THEN
      DELETE
        FROM openestate_objects_geodbs_areas
        WHERE object_id = val_object_id
        AND area_id = val_area_id;

    ELSE
      DELETE
        FROM openestate_objects_geodbs_areas
        WHERE object_id = val_object_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_geodbs_areas
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_objects_i18n
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects_i18n CASCADE;
\.
CREATE PROCEDURE remove_openestate_objects_i18n(
    val_object_id BIGINT,
    val_language_id BIGINT
  )
  SPECIFIC remove_openestate_objects_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzungen zu einer Sprache entfernen
    IF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_i18n
        WHERE object_id = val_object_id
        AND language_id = val_language_id;

    -- Alle Übersetzungen entfernen
    ELSE

      DELETE
        FROM openestate_objects_i18n
        WHERE object_id = val_object_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_i18n
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_objects_links
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects_links CASCADE;
\.
CREATE PROCEDURE remove_openestate_objects_links( val_link_id BIGINT )
  SPECIFIC remove_openestate_objects_links
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_link_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_links
      WHERE link_id = val_link_id;
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Datei entfernen
    DELETE
      FROM openestate_objects_links
      WHERE link_id = val_link_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_links
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_objects_links_i18n
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_objects_links_i18n CASCADE;
\.
CREATE PROCEDURE remove_openestate_objects_links_i18n(
    val_link_id BIGINT,
    val_language_id BIGINT
  )
  SPECIFIC remove_openestate_objects_links_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_link_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_links
      WHERE link_id = val_link_id;
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzungen zu einer Sprache entfernen
    IF val_language_id IS NOT NULL THEN

      DELETE
        FROM openestate_objects_links_i18n
        WHERE link_id = val_link_id
        AND language_id = val_language_id;

    -- Alle Übersetzungen entfernen
    ELSE

      DELETE
        FROM openestate_objects_links_i18n
        WHERE link_id = val_link_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_objects_links_i18n
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_searches
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_searches CASCADE;
\.
CREATE PROCEDURE remove_openestate_searches( val_search_id BIGINT )
  SPECIFIC remove_openestate_searches
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lösch-Rechte besitzt
    SET allowed = can_delete_immotool_agency_searches( val_search_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Suche entfernen
    DELETE
      FROM openestate_searches
      WHERE search_id = val_search_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_searches
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_searches_attribs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_searches_attribs CASCADE;
\.
CREATE PROCEDURE remove_openestate_searches_attribs( val_search_id BIGINT )
  SPECIFIC remove_openestate_searches_attribs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_searches( val_search_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Kriterien entfernen
    DELETE
      FROM openestate_searches_attribs
      WHERE search_id = val_search_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_searches_attribs
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_searches_params
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_searches_params CASCADE;
\.
CREATE PROCEDURE remove_openestate_searches_params(
    val_search_id BIGINT,
    val_param_key VARCHAR(255)
  )
  SPECIFIC remove_openestate_searches_params
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_searches( val_search_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Spezifischen Parameter entfernen
    IF val_param_key IS NOT NULL THEN

      DELETE
        FROM openestate_searches_params
        WHERE search_id = val_search_id
        AND param_key = val_param_key;

    -- Alle Parameter entfernen
    ELSE

      DELETE
        FROM openestate_searches_params
        WHERE search_id = val_search_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_searches_params
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure remove_openestate_searches_values
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS remove_openestate_searches_values CASCADE;
\.
CREATE PROCEDURE remove_openestate_searches_values( val_search_id BIGINT )
  SPECIFIC remove_openestate_searches_values
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_searches( val_search_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Kriterien entfernen
    DELETE
      FROM openestate_searches_values
      WHERE search_id = val_search_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE remove_openestate_searches_values
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_contacts
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_contacts CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_contacts(
    val_contact_id BIGINT,
    val_contact_enabled BOOLEAN,
    val_contact_notes LONGVARCHAR,
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_agency_contacts
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_id BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Datensatz ermitteln
    SELECT contact_id, access_permissions, access_owner_id, access_group_id
      INTO current_id, permissions, owner_uid, owner_gid
      FROM immotool_agency_contacts
      WHERE contact_id = val_contact_id;

    -- Datensatz existiert noch nicht
    IF current_id IS NULL THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Ansprechpartner-Zusatz einfügen
      INSERT INTO immotool_agency_contacts (
        contact_id,
        contact_enabled,
        contact_notes,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_contact_id,
        val_contact_enabled,
        val_contact_notes,
        owner_uid,
        owner_gid,
        permissions );

    -- Datensatz existiert bereits
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>1 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>1 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>1 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Ansprechpartner-Zusatz aktualisieren
      UPDATE immotool_agency_contacts
        SET
          contact_enabled = val_contact_enabled,
          contact_notes = val_contact_notes,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions
        WHERE
          contact_id = val_contact_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_contacts
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exports
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exports CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exports(
    INOUT val_export_id BIGINT,
    val_export_name VARCHAR(100),
    val_export_enabled BOOLEAN,
    val_export_limit INT,
    val_export_last_execution TIMESTAMP WITH TIME ZONE,
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_agency_exports
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Neuen Export erzeugen
    IF val_export_id < 1 THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Export speichern
      INSERT INTO immotool_agency_exports (
        export_name,
        export_enabled,
        export_limit,
        export_last_execution,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_export_name,
        val_export_enabled,
        val_export_limit,
        val_export_last_execution,
        owner_uid,
        owner_gid,
        permissions );

      -- ID zurückliefern
      SET val_export_id = IDENTITY();

    -- Bestehenden Export bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_agency_exports
        WHERE export_id = val_export_id;

      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>1 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>1 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>1 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Export bearbeiten
      UPDATE immotool_agency_exports
        SET
          export_name = val_export_name,
          export_enabled = val_export_enabled,
          export_limit = val_export_limit,
          export_last_execution = val_export_last_execution,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions,
          modified_at = NOW()
        WHERE
          export_id = val_export_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exports
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exports_entries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exports_entries CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exports_entries(
    val_export_id BIGINT,
    val_entry_object_id BIGINT,
    val_entry_object_nr VARCHAR(255),
    val_entry_action VARCHAR(10)
  )
  SPECIFIC save_immotool_agency_exports_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte auf einem der Objekte besitzt
    IF NOT allowed = TRUE AND val_entry_object_id IS NOT NULL AND val_entry_object_id > 0 THEN
      SET allowed = can_write_immotool_agency_objects( val_entry_object_id );
    END IF;
    IF NOT allowed = TRUE AND val_export_id IS NOT NULL AND val_export_id > 0 THEN
      SET allowed = can_write_immotool_agency_exports( val_export_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- ggf. existierenden Eintrag entfernen
    DELETE
      FROM immotool_agency_exports_entries
      WHERE export_id = val_export_id
      AND entry_object_id = val_entry_object_id;

    -- Eintrag speichern
    INSERT INTO immotool_agency_exports_entries (
      export_id,
      entry_object_id,
      entry_object_nr,
      entry_action )
    VALUES (
      val_export_id,
      val_entry_object_id,
      val_entry_object_nr,
      val_entry_action );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exports_entries
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exports_logs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exports_logs CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exports_logs(
    INOUT val_log_id BIGINT,
    val_export_id BIGINT,
    val_log_success BOOLEAN,
    val_log_incremental BOOLEAN,
    val_log_format VARCHAR(100),
    val_log_transport VARCHAR(100),
    val_log_limit INT,
    val_log_count_updates INT,
    val_log_count_deletes INT,
    val_log_count_errors INT,
    val_log_time TIMESTAMP WITH TIME ZONE
  )
  SPECIFIC save_immotool_agency_exports_logs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_export_id IS NOT NULL AND val_export_id > 0 THEN
      SET allowed = can_write_immotool_agency_exports( val_export_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neues Log
    IF val_log_id < 1 THEN

      -- Log speichern
      INSERT INTO immotool_agency_exports_logs (
        export_id,
        log_success,
        log_incremental,
        log_format,
        log_transport,
        log_limit,
        log_count_updates,
        log_count_deletes,
        log_count_errors,
        log_time )
      VALUES (
        val_export_id,
        val_log_success,
        val_log_incremental,
        val_log_format,
        val_log_transport,
        val_log_limit,
        val_log_count_updates,
        val_log_count_deletes,
        val_log_count_errors,
        val_log_time );

      -- ID zurückliefern
      SET val_log_id = IDENTITY();

    ELSE

      -- Log bearbeiten
      UPDATE immotool_agency_exports_logs
        SET
          export_id = val_export_id,
          log_success = val_log_success,
          log_incremental = val_log_incremental,
          log_format = val_log_format,
          log_transport = val_log_transport,
          log_limit = val_log_limit,
          log_count_updates = val_log_count_updates,
          log_count_deletes = val_log_count_deletes,
          log_count_errors = val_log_count_errors,
          log_time = val_log_time
        WHERE
          log_id = val_log_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exports_logs
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exports_logs_entries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exports_logs_entries CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exports_logs_entries(
    val_log_id BIGINT,
    val_entry_object_id BIGINT,
    val_entry_object_nr VARCHAR(255),
    val_entry_action VARCHAR(10),
    val_entry_exported BOOLEAN,
    val_entry_message LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_exports_logs_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_export_id BIGINT;
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SELECT export_id
      INTO val_export_id
      FROM immotool_agency_exports_logs
      WHERE log_id = val_log_id;
    IF NOT allowed = TRUE AND val_export_id IS NOT NULL AND val_export_id > 0 THEN
      SET allowed = can_write_immotool_agency_exports( val_export_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- ggf. existierenden Eintrag entfernen
    DELETE
      FROM immotool_agency_exports_logs_entries
      WHERE log_id = val_log_id
      AND entry_object_id = val_entry_object_id;

    -- Eintrag speichern
    INSERT INTO immotool_agency_exports_logs_entries (
      log_id,
      entry_object_id,
      entry_object_nr,
      entry_action,
      entry_exported,
      entry_message )
    VALUES (
      val_log_id,
      val_entry_object_id,
      val_entry_object_nr,
      val_entry_action,
      val_entry_exported,
      val_entry_message );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exports_logs_entries
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exports_publishings
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exports_publishings CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exports_publishings(
    val_export_id BIGINT,
    val_publishing_object_id BIGINT,
    val_publishing_object_nr VARCHAR(255),
    val_publishing_object_key VARCHAR(255),
    val_publishing_time TIMESTAMP WITH TIME ZONE
  )
  SPECIFIC save_immotool_agency_exports_publishings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_export_id IS NOT NULL AND val_export_id > 0 THEN
      SET allowed = can_write_immotool_agency_exports( val_export_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- ggf. existierenden Eintrag entfernen
    DELETE
      FROM immotool_agency_exports_publishings
      WHERE export_id = val_export_id
      AND publishing_object_id = val_publishing_object_id;

    -- Eintrag speichern
    INSERT INTO immotool_agency_exports_publishings (
      export_id,
      publishing_object_id,
      publishing_object_nr,
      publishing_object_key,
      publishing_time )
    VALUES (
      val_export_id,
      val_publishing_object_id,
      val_publishing_object_nr,
      val_publishing_object_key,
      val_publishing_time );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exports_publishings
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exports_settings
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exports_settings CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exports_settings(
    val_export_id BIGINT,
    val_setting_key VARCHAR(255),
    val_setting_value LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_exports_settings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_export_id IS NOT NULL AND val_export_id > 0 THEN
      SET allowed = can_write_immotool_agency_exports( val_export_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- ggf. existierendes Setting entfernen
    DELETE
      FROM immotool_agency_exports_settings
      WHERE export_id = val_export_id
      AND setting_key = val_setting_key;

    -- Setting speichern
    INSERT INTO immotool_agency_exports_settings (
      export_id,
      setting_key,
      setting_value )
    VALUES (
      val_export_id,
      val_setting_key,
      val_setting_value );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exports_settings
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exposes
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exposes CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exposes(
    INOUT val_expose_id BIGINT,
    val_expose_name VARCHAR(100),
    val_expose_type VARCHAR(255),
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_agency_exposes
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Neues Exposé erzeugen
    IF val_expose_id < 1 THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Exposé speichern
      INSERT INTO immotool_agency_exposes (
        expose_name,
        expose_type,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_expose_name,
        val_expose_type,
        owner_uid,
        owner_gid,
        permissions );

      -- ID zurückliefern
      SET val_expose_id = IDENTITY();

    -- Bestehendes Exposé bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SELECT access_permissions, access_owner_id, access_group_id
        INTO permissions, owner_uid, owner_gid
        FROM immotool_agency_exposes
        WHERE expose_id = val_expose_id;

      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>1 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>1 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>1 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Exposé bearbeiten
      UPDATE immotool_agency_exposes
        SET
          expose_name = val_expose_name,
          expose_type = val_expose_type,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions,
          modified_at = NOW()
        WHERE
          expose_id = val_expose_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exposes
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exposes_files
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exposes_files CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exposes_files(
    INOUT val_file_id BIGINT,
    val_expose_id BIGINT,
    val_file_name VARCHAR(255),
    val_file_mimetype VARCHAR(255),
    val_file_content BLOB(1G)
  )
  SPECIFIC save_immotool_agency_exposes_files
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_expose_id IS NOT NULL AND val_expose_id > 0 THEN
      SET allowed = can_write_immotool_agency_exposes( val_expose_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neue Datei
    IF val_file_id < 1 THEN

      -- Datei speichern
      INSERT INTO immotool_agency_exposes_files (
        expose_id,
        file_name,
        file_mimetype )
      VALUES (
        val_expose_id,
        val_file_name,
        val_file_mimetype );

      -- ID zurückliefern
      SET val_file_id = IDENTITY();

    ELSE

      -- Datei bearbeiten
      UPDATE immotool_agency_exposes_files
        SET
          expose_id = val_expose_id,
          file_name = val_file_name,
          file_mimetype = val_file_mimetype,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

    -- Datei ggf. aktualisieren
    IF val_file_content IS NOT NULL THEN

      -- Datei bearbeiten
      UPDATE immotool_agency_exposes_files
        SET
          file_content = val_file_content,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exposes_files
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_exposes_settings
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_exposes_settings CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_exposes_settings(
    val_expose_id BIGINT,
    val_setting_key VARCHAR(255),
    val_setting_value LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_exposes_settings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_expose_id IS NOT NULL AND val_expose_id > 0 THEN
      SET allowed = can_write_immotool_agency_exposes( val_expose_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- ggf. existierendes Setting entfernen
    DELETE
      FROM immotool_agency_exposes_settings
      WHERE expose_id = val_expose_id
      AND setting_key = val_setting_key;

    -- Setting speichern
    INSERT INTO immotool_agency_exposes_settings (
      expose_id,
      setting_key,
      setting_value )
    VALUES (
      val_expose_id,
      val_setting_key,
      val_setting_value );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_exposes_settings
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_objects
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_objects CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_objects(
    val_object_id BIGINT,
    val_object_status VARCHAR(25),
    val_object_export_mode VARCHAR(25),
    val_object_import_id VARCHAR(255),
    val_object_import_hash VARCHAR(255),
    val_object_group_nr INT,
    val_object_notes LONGVARCHAR,
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_agency_objects
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_id BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Datensatz ermitteln
    SELECT object_id, access_permissions, access_owner_id, access_group_id
      INTO current_id, permissions, owner_uid, owner_gid
      FROM immotool_agency_objects
      WHERE object_id = val_object_id;

    -- Datensatz existiert noch nicht
    IF current_id IS NULL THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Immobilien-Zusatz einfügen
      INSERT INTO immotool_agency_objects (
        object_id,
        object_status,
        object_export_mode,
        object_import_id,
        object_import_hash,
        object_group_nr,
        object_notes,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_object_id,
        val_object_status,
        val_object_export_mode,
        val_object_import_id,
        val_object_import_hash,
        val_object_group_nr,
        val_object_notes,
        owner_uid,
        owner_gid,
        permissions );

    -- Datensatz existiert bereits
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>1 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>1 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>1 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Immobilien-Zusatz aktualisieren
      UPDATE immotool_agency_objects
        SET
          object_status = val_object_status,
          object_export_mode = val_object_export_mode,
          object_import_id = val_object_import_id,
          object_import_hash = val_object_import_hash,
          object_group_nr = val_object_group_nr,
          object_notes = val_object_notes,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions
        WHERE
          object_id = val_object_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_objects
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_objects_exports
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_objects_exports CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_objects_exports(
    val_object_id BIGINT,
    val_export_id BIGINT
  )
  SPECIFIC save_immotool_agency_objects_exports
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id IS NULL OR val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;
    IF val_export_id IS NULL OR val_export_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte auf einem der Objekte besitzt
    IF NOT allowed = TRUE THEN
      SET allowed = can_write_immotool_agency_objects( val_object_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SET allowed = can_write_immotool_agency_exports( val_export_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Verknüpfung ggf. entfernen
    DELETE
      FROM immotool_agency_objects_exports
      WHERE object_id = val_object_id
      AND export_id = val_export_id;

    -- Verknüpfung speichern
    INSERT INTO immotool_agency_objects_exports (
      object_id,
      export_id )
    VALUES (
      val_object_id,
      val_export_id );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_objects_exports
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_objects_logs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_objects_logs CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_objects_logs(
    val_log_id BIGINT,
    val_object_id BIGINT
  )
  SPECIFIC save_immotool_agency_objects_logs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_objects( val_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Protokolleintrag aktualisieren
    DELETE
      FROM immotool_agency_objects_logs
      WHERE log_id = val_log_id;

    INSERT
      INTO immotool_agency_objects_logs (log_id, object_id)
      VALUES (val_log_id, val_object_id);

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_objects_logs
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_immotool_agency_searches
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_immotool_agency_searches CASCADE;
\.
CREATE PROCEDURE save_immotool_agency_searches(
    val_search_id BIGINT,
    val_search_sticky BOOLEAN,
    val_search_object_status VARCHAR(255),
    val_access_owner_id BIGINT,
    val_access_group_id BIGINT,
    val_access_permissions INT
  )
  SPECIFIC save_immotool_agency_searches
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;
    DECLARE current_id BIGINT;
    DECLARE current_uid BIGINT;
    DECLARE dba BOOLEAN;
    SET dba = is_admin();

    SELECT user_id
      INTO current_uid
      FROM view_immotool_users
      WHERE user_login=USER();

    -- Datensatz ermitteln
    SELECT search_id, access_permissions, access_owner_id, access_group_id
      INTO current_id, permissions, owner_uid, owner_gid
      FROM immotool_agency_searches
      WHERE search_id = val_search_id;

    -- Datensatz existiert noch nicht
    IF current_id IS NULL THEN
      -- Inhaber-Benutzer ggf. automatisch setzen
      IF (val_access_owner_id IS NULL OR val_access_owner_id<1 OR dba = FALSE) THEN
        SET owner_uid = current_uid;
      ELSE
        SET owner_uid = val_access_owner_id;
      END IF;

      -- Inhaber-Gruppe ggf. automatisch setzen
      IF (val_access_group_id IS NULL OR val_access_group_id<1) THEN
        SELECT group_id
          INTO owner_gid
          FROM view_immotool_groups
          WHERE group_name='IMMOTOOL';
      ELSE
        SET owner_gid = val_access_group_id;
      END IF;

      -- Berechtigungen ggf. automatisch setzen
      IF (val_access_permissions IS NULL OR val_access_permissions<0) THEN
        SET permissions = 63;
      ELSE
        SET permissions = val_access_permissions;
      END IF;

      -- Suche-Zusatz einfügen
      INSERT INTO immotool_agency_searches (
        search_id,
        search_sticky,
        search_object_status,
        access_owner_id,
        access_group_id,
        access_permissions )
      VALUES (
        val_search_id,
        val_search_sticky,
        val_search_object_status,
        owner_uid,
        owner_gid,
        permissions );

    -- Datensatz existiert bereits
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      IF dba = FALSE THEN
        SET allowed = can_write(permissions, owner_uid, owner_gid);
        IF NOT allowed = TRUE THEN
          SIGNAL SQLSTATE '45000';
        END IF;
      END IF;

      -- Wechsel des Inhaber-Benutzers darf nur der Administrator durchführen
      IF dba = TRUE AND val_access_owner_id IS NOT NULL AND val_access_owner_id>1 THEN
        SET owner_uid = val_access_owner_id;
      END IF;

      IF dba = TRUE OR owner_uid = current_uid THEN

        -- Wechsel der Inhaber-Gruppe darf nur der Administrator oder Inhaber durchführen
        IF val_access_group_id IS NOT NULL AND val_access_group_id>1 THEN
          SET owner_gid = val_access_group_id;
        END IF;

        -- Wechsel der Berechtigungen darf nur der Administrator oder Inhaber durchführen
        IF val_access_permissions IS NOT NULL AND val_access_permissions>1 THEN
          SET permissions = val_access_permissions;
        END IF;
      END IF;

      -- Suche-Zusatz aktualisieren
      UPDATE immotool_agency_searches
        SET
          search_sticky = val_search_sticky,
          search_object_status = val_search_object_status,
          access_owner_id = owner_uid,
          access_group_id = owner_gid,
          access_permissions = permissions
        WHERE
          search_id = val_search_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_searches
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_contacts
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_contacts CASCADE;
\.
CREATE PROCEDURE save_openestate_contacts(
    INOUT val_contact_id BIGINT,
    val_person_first_name VARCHAR(150),
    val_person_middle_name VARCHAR(150),
    val_person_last_name VARCHAR(150),
    val_person_title VARCHAR(50),
    val_person_gender CHAR(1),
    val_person_date_of_birth DATE,
    val_person_mail VARCHAR(50),
    val_person_phone VARCHAR(50),
    val_person_mobile_phone VARCHAR(50),
    val_person_fax VARCHAR(50),
    val_address_street VARCHAR(150),
    val_address_street2 VARCHAR(150),
    val_address_street_nr VARCHAR(20),
    val_address_postal_code VARCHAR(10),
    val_address_city VARCHAR(150),
    val_address_district VARCHAR(150),
    val_address_country CHAR(2),
    val_address_region VARCHAR(150),
    val_address_latitude DECIMAL(15,12),
    val_address_longitude DECIMAL(15,12)
  )
  SPECIFIC save_openestate_contacts
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Neuer Ansprechpartner erzeugen
    IF val_contact_id < 1 THEN

      -- Ansprechpartner speichern
      INSERT INTO openestate_contacts (
        person_first_name,
        person_middle_name,
        person_last_name,
        person_title,
        person_gender,
        person_date_of_birth,
        person_mail,
        person_phone,
        person_mobile_phone,
        person_fax,
        address_street,
        address_street2,
        address_street_nr,
        address_postal_code,
        address_city,
        address_district,
        address_country,
        address_region,
        address_latitude,
        address_longitude )
      VALUES (
        val_person_first_name,
        val_person_middle_name,
        val_person_last_name,
        val_person_title,
        val_person_gender,
        val_person_date_of_birth,
        val_person_mail,
        val_person_phone,
        val_person_mobile_phone,
        val_person_fax,
        val_address_street,
        val_address_street2,
        val_address_street_nr,
        val_address_postal_code,
        val_address_city,
        val_address_district,
        val_address_country,
        val_address_region,
        val_address_latitude,
        val_address_longitude );

      -- ID zurückliefern
      SET val_contact_id = IDENTITY();

    -- Bestehenden Ansprechpartner bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SET allowed = can_write_immotool_agency_contacts(val_contact_id);
      IF NOT allowed = TRUE THEN
        SIGNAL SQLSTATE '45000';
      END IF;

      -- Ansprechpartner bearbeiten
      UPDATE openestate_contacts
        SET
          person_first_name = val_person_first_name,
          person_middle_name = val_person_middle_name,
          person_last_name = val_person_last_name,
          person_title = val_person_title,
          person_gender = val_person_gender,
          person_date_of_birth = val_person_date_of_birth,
          person_mail = val_person_mail,
          person_phone = val_person_phone,
          person_mobile_phone = val_person_mobile_phone,
          person_fax = val_person_fax,
          address_street = val_address_street,
          address_street2 = val_address_street2,
          address_street_nr = val_address_street_nr,
          address_postal_code = val_address_postal_code,
          address_city = val_address_city,
          address_district = val_address_district,
          address_country = val_address_country,
          address_region = val_address_region,
          address_latitude = val_address_latitude,
          address_longitude = val_address_longitude,
          modified_at = NOW()
        WHERE
          contact_id = val_contact_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_contacts
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_geodbs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_geodbs CASCADE;
\.
CREATE PROCEDURE save_openestate_geodbs(
    INOUT val_geodb_id BIGINT,
    val_geodb_name VARCHAR(255),
    val_geodb_provider VARCHAR(255),
    val_geodb_build_time TIMESTAMP WITH TIME ZONE
  )
  SPECIFIC save_openestate_geodbs
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- neue Geo-DB erzeugen
    IF val_geodb_id < 1 THEN
      INSERT INTO openestate_geodbs (
        geodb_name,
        geodb_provider,
        geodb_build_time )
      VALUES (
        val_geodb_name,
        val_geodb_provider,
        val_geodb_build_time );

      -- ID zurückliefern
      SET val_geodb_id = IDENTITY();

    -- bestehende Geo-DB bearbeiten
    ELSE
      UPDATE openestate_geodbs
        SET
          geodb_name = val_geodb_name,
          geodb_provider = val_geodb_provider,
          geodb_build_time = val_geodb_build_time,
          modified_at = NOW()
        WHERE
          geodb_id = val_geodb_id;

    END IF;

  END;
.;

-- -----------------------------------------------------
-- Procedure save_openestate_geodbs_areas
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_geodbs_areas CASCADE;
\.
CREATE PROCEDURE save_openestate_geodbs_areas(
    INOUT val_area_id BIGINT,
    val_country_id BIGINT,
    val_area_parent_id BIGINT,
    val_area_name VARCHAR(255),
    val_area_level VARCHAR(15),
    val_area_provider_id VARCHAR(255)
  )
  SPECIFIC save_openestate_geodbs_areas
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- neue Region erzeugen
    IF val_area_id < 1 THEN
      INSERT INTO openestate_geodbs_areas (
        country_id,
        area_parent_id,
        area_name,
        area_level,
        area_provider_id )
      VALUES (
        val_country_id,
        val_area_parent_id,
        val_area_name,
        val_area_level,
        val_area_provider_id );

      -- ID zurückliefern
      SET val_area_id = IDENTITY();

    -- bestehende Region bearbeiten
    ELSE
      UPDATE openestate_geodbs_areas
        SET
          country_id = val_country_id,
          area_parent_id = val_area_parent_id,
          area_name = val_area_name,
          area_level = val_area_level,
          area_provider_id = val_area_provider_id
        WHERE
          area_id = val_area_id;

    END IF;

  END;
.;

-- -----------------------------------------------------
-- Procedure save_openestate_geodbs_countries
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_geodbs_countries CASCADE;
\.
CREATE PROCEDURE save_openestate_geodbs_countries(
    INOUT val_country_id BIGINT,
    val_geodb_id BIGINT,
    val_country_iso CHAR(2),
    val_country_name VARCHAR(255),
    val_country_provider_id VARCHAR(255)
  )
  SPECIFIC save_openestate_geodbs_countries
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- neues Land erzeugen
    IF val_country_id < 1 THEN
      INSERT INTO openestate_geodbs_countries (
        geodb_id,
        country_iso,
        country_name,
        country_provider_id )
      VALUES (
        val_geodb_id,
        val_country_iso,
        val_country_name,
        val_country_provider_id );

      -- ID zurückliefern
      SET val_country_id = IDENTITY();

    -- bestehendes Land bearbeiten
    ELSE
      UPDATE openestate_geodbs_countries
        SET
          geodb_id = val_geodb_id,
          country_iso = val_country_iso,
          country_name = val_country_name,
          country_provider_id = val_country_provider_id
        WHERE
          country_id = val_country_id;

    END IF;

  END;
.;

-- -----------------------------------------------------
-- Procedure save_openestate_languages
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_languages CASCADE;
\.
CREATE PROCEDURE save_openestate_languages(
    INOUT val_language_id BIGINT,
    val_language_code VARCHAR(50),
    val_language_enabled BOOLEAN
  )
  SPECIFIC save_openestate_languages
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Neue Sprache
    IF val_language_id < 1 THEN

      -- Sprache speichern
      INSERT INTO openestate_languages (
        language_code,
        language_enabled )
      VALUES (
        val_language_code,
        val_language_enabled );

      -- ID zurückliefern
      SET val_language_id = IDENTITY();

    ELSE

      -- Sprache bearbeiten
      UPDATE openestate_languages
        SET
          language_code = val_language_code,
          language_enabled = val_language_enabled,
          modified_at = NOW()
        WHERE
          language_id = val_language_id;

    END IF;

  END;
.;

-- -----------------------------------------------------
-- Procedure save_openestate_objects
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_objects CASCADE;
\.
CREATE PROCEDURE save_openestate_objects(
    INOUT val_object_id BIGINT,
    val_contact_id BIGINT,
    val_object_type VARCHAR(100),
    val_object_action VARCHAR(100),
    val_object_public_id VARCHAR(255),
    val_object_currency CHAR(3),
    val_object_hidden_prices BOOLEAN,
    val_object_publish_begin TIMESTAMP,
    val_object_publish_end TIMESTAMP,
    val_address_published BOOLEAN,
    val_address_street VARCHAR(150),
    val_address_street2 VARCHAR(150),
    val_address_street_nr VARCHAR(20),
    val_address_postal_code VARCHAR(10),
    val_address_city VARCHAR(150),
    val_address_district VARCHAR(150),
    val_address_country CHAR(2),
    val_address_region VARCHAR(150),
    val_address_latitude DECIMAL(15,12),
    val_address_longitude DECIMAL(15,12)
  )
  SPECIFIC save_openestate_objects
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Neues Objekt erzeugen
    IF val_object_id < 1 THEN

      -- Objekt speichern
      INSERT INTO openestate_objects (
        contact_id,
        object_type,
        object_action,
        object_public_id,
        object_currency,
        object_hidden_prices,
        object_publish_begin,
        object_publish_end,
        address_published,
        address_street,
        address_street2,
        address_street_nr,
        address_postal_code,
        address_city,
        address_district,
        address_country,
        address_region,
        address_latitude,
        address_longitude )
      VALUES (
        val_contact_id,
        val_object_type,
        val_object_action,
        val_object_public_id,
        val_object_currency,
        val_object_hidden_prices,
        val_object_publish_begin,
        val_object_publish_end,
        val_address_published,
        val_address_street,
        val_address_street2,
        val_address_street_nr,
        val_address_postal_code,
        val_address_city,
        val_address_district,
        val_address_country,
        val_address_region,
        val_address_latitude,
        val_address_longitude );

      -- ID zurückliefern
      SET val_object_id = IDENTITY();

    -- Bestehendes Objekt bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SET allowed = can_write_immotool_agency_objects(val_object_id);
      IF NOT allowed = TRUE THEN
        SIGNAL SQLSTATE '45000';
      END IF;

      -- Objekt bearbeiten
      UPDATE openestate_objects
        SET
          contact_id = val_contact_id,
          object_type = val_object_type,
          object_action = val_object_action,
          object_public_id = val_object_public_id,
          object_currency = val_object_currency,
          object_hidden_prices = val_object_hidden_prices,
          object_publish_begin = val_object_publish_begin,
          object_publish_end = val_object_publish_end,
          address_published = val_address_published,
          address_street = val_address_street,
          address_street2 = val_address_street2,
          address_street_nr = val_address_street_nr,
          address_postal_code = val_address_postal_code,
          address_city = val_address_city,
          address_district = val_address_district,
          address_country = val_address_country,
          address_region = val_address_region,
          address_latitude = val_address_latitude,
          address_longitude = val_address_longitude,
          modified_at = NOW()
        WHERE
          object_id = val_object_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_objects
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_objects_attribs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_objects_attribs CASCADE;
\.
CREATE PROCEDURE save_openestate_objects_attribs(
    val_object_id BIGINT,
    val_language_id BIGINT,
    val_attrib_group VARCHAR(255),
    val_attrib_name VARCHAR(255),
    val_attrib_unit VARCHAR(25),
    val_value_text LONGVARCHAR,
    val_value_string VARCHAR(255),
    val_value_boolean BOOLEAN,
    val_value_numeric DECIMAL(15,2),
    val_value_numeric_base DECIMAL(15,2),
    val_value_date DATE,
    val_value_time TIME,
    val_value_datetime TIMESTAMP WITH TIME ZONE
  )
  SPECIFIC save_openestate_objects_attribs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Attribut speichern
    INSERT INTO openestate_objects_attribs (
      object_id,
      language_id,
      attrib_group,
      attrib_name,
      attrib_unit,
      value_text,
      value_string,
      value_boolean,
      value_numeric,
      value_numeric_base,
      value_date,
      value_time,
      value_datetime )
    VALUES (
      val_object_id,
      val_language_id,
      val_attrib_group,
      val_attrib_name,
      val_attrib_unit,
      val_value_text,
      val_value_string,
      val_value_boolean,
      val_value_numeric,
      val_value_numeric_base,
      val_value_date,
      val_value_time,
      val_value_datetime );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_attribs
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_objects_files
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_objects_files CASCADE;
\.
CREATE PROCEDURE save_openestate_objects_files(
    INOUT val_file_id BIGINT,
    val_object_id BIGINT,
    val_file_name VARCHAR(255),
    val_file_url VARCHAR(255),
    val_file_content BLOB(1G),
    val_file_content_preview BLOB(1M),
    val_file_mimetype VARCHAR(50),
    val_file_published BOOLEAN,
    val_file_type VARCHAR(255),
    val_file_size BIGINT,
    val_file_image_width INT,
    val_file_image_height INT,
    val_file_position INT
  )
  SPECIFIC save_openestate_objects_files
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neue Datei
    IF val_file_id < 1 THEN

      -- Datei speichern
      INSERT INTO openestate_objects_files (
        object_id,
        file_name,
        file_url,
        file_mimetype,
        file_published,
        file_type,
        file_size,
        file_image_width,
        file_image_height,
        file_position )
      VALUES (
        val_object_id,
        val_file_name,
        val_file_url,
        val_file_mimetype,
        val_file_published,
        val_file_type,
        val_file_size,
        val_file_image_width,
        val_file_image_height,
        val_file_position );

      -- ID zurückliefern
      SET val_file_id = IDENTITY();

    ELSE

      -- Datei bearbeiten
      UPDATE openestate_objects_files
        SET
          object_id = val_object_id,
          file_name = val_file_name,
          file_url = val_file_url,
          file_mimetype = val_file_mimetype,
          file_published = val_file_published,
          file_type = val_file_type,
          file_size = val_file_size,
          file_image_width = val_file_image_width,
          file_image_height = val_file_image_height,
          file_position = val_file_position,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

    -- Datei ggf. aktualisieren
    IF val_file_content IS NOT NULL THEN

      -- Datei bearbeiten
      UPDATE openestate_objects_files
        SET
          file_content = val_file_content,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

    -- Vorschau ggf. aktualisieren
    IF val_file_content_preview IS NOT NULL THEN

      -- Vorschau bearbeiten
      UPDATE openestate_objects_files
        SET
          file_content_preview = val_file_content_preview,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_files
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_objects_files_i18n
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_objects_files_i18n CASCADE;
\.
CREATE PROCEDURE save_openestate_objects_files_i18n(
    val_file_id BIGINT,
    val_language_id BIGINT,
    val_i18n_title VARCHAR(255)
  )
  SPECIFIC save_openestate_objects_files_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_files
      WHERE file_id = val_file_id;
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzung ggf. entfernen
    DELETE
      FROM openestate_objects_files_i18n
      WHERE file_id = val_file_id
      AND language_id = val_language_id;

    -- Übersetzung speichern
    INSERT INTO openestate_objects_files_i18n (
      file_id,
      language_id,
      i18n_title )
    VALUES (
      val_file_id,
      val_language_id,
      val_i18n_title );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_files_i18n
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_objects_geodbs_areas
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_objects_geodbs_areas CASCADE;
\.
CREATE PROCEDURE save_openestate_objects_geodbs_areas(
    val_object_id BIGINT,
    val_area_id BIGINT
  )
  SPECIFIC save_openestate_objects_geodbs_areas
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    DELETE
      FROM openestate_objects_geodbs_areas
      WHERE object_id = val_object_id
      AND area_id = val_area_id;

    INSERT
      INTO openestate_objects_geodbs_areas (
        object_id,
        area_id)
      VALUES (
        val_object_id,
        val_area_id);

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_geodbs_areas
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_objects_i18n
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_objects_i18n CASCADE;
\.
CREATE PROCEDURE save_openestate_objects_i18n(
    val_object_id BIGINT,
    val_language_id BIGINT,
    val_i18n_title VARCHAR(255)
  )
  SPECIFIC save_openestate_objects_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzung ggf. entfernen
    DELETE
      FROM openestate_objects_i18n
      WHERE object_id = val_object_id
      AND language_id = val_language_id;

    -- Übersetzung speichern
    INSERT INTO openestate_objects_i18n (
      object_id,
      language_id,
      i18n_title )
    VALUES (
      val_object_id,
      val_language_id,
      val_i18n_title );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_i18n
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_objects_links
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_objects_links CASCADE;
\.
CREATE PROCEDURE save_openestate_objects_links(
    INOUT val_link_id BIGINT,
    val_object_id BIGINT,
    val_link_web_url VARCHAR(255),
    val_link_provider_id VARCHAR(255),
    val_link_provider_name VARCHAR(255),
    val_link_published BOOLEAN,
    val_link_position INT
  )
  SPECIFIC save_openestate_objects_links
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuer Link
    IF val_link_id < 1 THEN

      -- Datei speichern
      INSERT INTO openestate_objects_links (
        object_id,
        link_web_url,
        link_provider_id,
        link_provider_name,
        link_published,
        link_position )
      VALUES (
        val_object_id,
        val_link_web_url,
        val_link_provider_id,
        val_link_provider_name,
        val_link_published,
        val_link_position );

      -- ID zurückliefern
      SET val_link_id = IDENTITY();

    ELSE

      -- Link bearbeiten
      UPDATE openestate_objects_links
        SET
          object_id = val_object_id,
          link_web_url = val_link_web_url,
          link_provider_id = val_link_provider_id,
          link_provider_name = val_link_provider_name,
          link_published = val_link_published,
          link_position = val_link_position,
          modified_at = NOW()
        WHERE
          link_id = val_link_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_links
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_objects_links_i18n
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_objects_links_i18n CASCADE;
\.
CREATE PROCEDURE save_openestate_objects_links_i18n(
    val_link_id BIGINT,
    val_language_id BIGINT,
    val_i18n_title VARCHAR(255)
  )
  SPECIFIC save_openestate_objects_links_i18n
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_link_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_links
      WHERE link_id = val_link_id;
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Übersetzung ggf. entfernen
    DELETE
      FROM openestate_objects_links_i18n
      WHERE link_id = val_link_id
      AND language_id = val_language_id;

    -- Übersetzung speichern
    INSERT INTO openestate_objects_links_i18n (
      link_id,
      language_id,
      i18n_title )
    VALUES (
      val_link_id,
      val_language_id,
      val_i18n_title );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_objects_links_i18n
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_searches
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_searches CASCADE;
\.
CREATE PROCEDURE save_openestate_searches(
    INOUT val_search_id BIGINT,
    val_search_name VARCHAR(255),
    val_search_notes LONGVARCHAR,
    val_search_enabled BOOLEAN
  )
  SPECIFIC save_openestate_searches
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- Neue Suche erzeugen
    IF val_search_id < 1 THEN

      -- Suche speichern
      INSERT INTO openestate_searches (
        search_name,
        search_notes,
        search_enabled )
      VALUES (
        val_search_name,
        val_search_notes,
        val_search_enabled );

      -- ID zurückliefern
      SET val_search_id = IDENTITY();

    -- Bestehende Suche bearbeiten
    ELSE
      -- Prüfen ob Schreib-Rechte vorliegen
      SET allowed = can_write_immotool_agency_searches(val_search_id);
      IF NOT allowed = TRUE THEN
        SIGNAL SQLSTATE '45000';
      END IF;

      -- Suche bearbeiten
      UPDATE openestate_searches
        SET
          search_name = val_search_name,
          search_notes = val_search_notes,
          search_enabled = val_search_enabled,
          modified_at = NOW()
        WHERE
          search_id = val_search_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_searches
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_searches_attribs
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_searches_attribs CASCADE;
\.
CREATE PROCEDURE save_openestate_searches_attribs(
    val_search_id BIGINT,
    val_attrib_group VARCHAR(255),
    val_attrib_name VARCHAR(255),
    val_attrib_equals VARCHAR(255),
    val_attrib_equals_like BOOLEAN,
    val_attrib_equals_case BOOLEAN,
    val_attrib_matches VARCHAR(255),
    val_attrib_numeric_min DECIMAL(15,2),
    val_attrib_numeric_max DECIMAL(15,2),
    val_attrib_numeric_unit VARCHAR(20),
    val_attrib_time_begin TIMESTAMP WITH TIME ZONE,
    val_attrib_time_end TIMESTAMP WITH TIME ZONE
  )
  SPECIFIC save_openestate_searches_attribs
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_searches(val_search_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Kriterium speichern
    INSERT INTO openestate_searches_attribs (
      search_id,
      attrib_group,
      attrib_name,
      attrib_equals,
      attrib_equals_like,
      attrib_equals_case,
      attrib_matches,
      attrib_numeric_min,
      attrib_numeric_max,
      attrib_numeric_unit,
      attrib_time_begin,
      attrib_time_end )
    VALUES (
      val_search_id,
      val_attrib_group,
      val_attrib_name,
      val_attrib_equals,
      val_attrib_equals_like,
      val_attrib_equals_case,
      val_attrib_matches,
      val_attrib_numeric_min,
      val_attrib_numeric_max,
      val_attrib_numeric_unit,
      val_attrib_time_begin,
      val_attrib_time_end );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_searches_attribs
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_searches_params
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_searches_params CASCADE;
\.
CREATE PROCEDURE save_openestate_searches_params(
    val_search_id BIGINT,
    val_param_key VARCHAR(255),
    val_param_value VARCHAR(255)
  )
  SPECIFIC save_openestate_searches_params
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_searches(val_search_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- existierenden Paramter ggf. entfernen
    IF val_param_key IS NOT NULL THEN

      DELETE
        FROM openestate_searches_params
        WHERE search_id = val_search_id
        AND param_key = val_param_key;

    END IF;

    -- Parameter speichern
    INSERT INTO openestate_searches_params (
      search_id,
      param_key,
      param_value )
    VALUES (
      val_search_id,
      val_param_key,
      val_param_value );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_searches_params
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure save_openestate_searches_values
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS save_openestate_searches_values CASCADE;
\.
CREATE PROCEDURE save_openestate_searches_values(
    val_search_id BIGINT,
    val_value_type VARCHAR(100),
    val_value_equals VARCHAR(255),
    val_value_equals_like BOOLEAN,
    val_value_equals_case BOOLEAN
  )
  SPECIFIC save_openestate_searches_values
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_searches(val_search_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Kriterium speichern
    INSERT INTO openestate_searches_values (
      search_id,
      value_type,
      value_equals,
      value_equals_like,
      value_equals_case )
    VALUES (
      val_search_id,
      val_value_type,
      val_value_equals,
      val_value_equals_like,
      val_value_equals_case );

  END;
.;
GRANT EXECUTE
  ON PROCEDURE save_openestate_searches_values
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure set_immotool_agency_exports_enabled
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS set_immotool_agency_exports_enabled CASCADE;
\.
CREATE PROCEDURE set_immotool_agency_exports_enabled(
    val_export_id BIGINT,
    val_export_enabled BOOLEAN
  )
  SPECIFIC set_immotool_agency_exports_enabled
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_export_id IS NOT NULL AND val_export_id > 0 THEN
      SET allowed = can_write_immotool_agency_exports( val_export_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Export bearbeiten
    UPDATE immotool_agency_exports
      SET
        export_enabled = val_export_enabled,
        modified_at = NOW()
      WHERE
        export_id = val_export_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE set_immotool_agency_exports_enabled
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure set_immotool_agency_objects_status
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS set_immotool_agency_objects_status CASCADE;
\.
CREATE PROCEDURE set_immotool_agency_objects_status(
    val_object_id BIGINT,
    val_object_status VARCHAR(25)
  )
  SPECIFIC set_immotool_agency_objects_status
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_object_id IS NOT NULL AND val_object_id > 0 THEN
      SET allowed = can_write_immotool_agency_objects( val_object_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Immobilie bearbeiten
    UPDATE immotool_agency_objects
      SET
        object_status = val_object_status
      WHERE
        object_id = val_object_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE set_immotool_agency_objects_status
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure set_openestate_languages_enabled
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS set_openestate_languages_enabled CASCADE;
\.
CREATE PROCEDURE set_openestate_languages_enabled(
    val_language_id BIGINT,
    val_language_enabled BOOLEAN
  )
  SPECIFIC set_openestate_languages_enabled
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_language_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Sprache bearbeiten
    UPDATE openestate_languages
      SET
        language_enabled = val_language_enabled,
        modified_at = NOW()
      WHERE
        language_id = val_language_id;

  END;
.;

-- -----------------------------------------------------
-- Procedure set_openestate_objects_contact
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS set_openestate_objects_contact CASCADE;
\.
CREATE PROCEDURE set_openestate_objects_contact(
    val_object_id BIGINT,
    val_contact_id BIGINT
  )
  SPECIFIC set_openestate_objects_contact
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    IF NOT allowed = TRUE AND val_object_id IS NOT NULL AND val_object_id > 0 THEN
      SET allowed = can_write_immotool_agency_objects( val_object_id );
    END IF;
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Objekt bearbeiten
    UPDATE openestate_objects
      SET
        contact_id = val_contact_id,
        modified_at = NOW()
      WHERE
        object_id = val_object_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE set_openestate_objects_contact
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure set_openestate_objects_files_content
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS set_openestate_objects_files_content CASCADE;
\.
CREATE PROCEDURE set_openestate_objects_files_content(
    val_file_id BIGINT,
    val_file_size BIGINT,
    val_file_content BLOB(1G),
    val_file_content_preview BLOB(1M)
  )
  SPECIFIC set_openestate_objects_files_content
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE val_object_id BIGINT;
    DECLARE allowed BOOLEAN;

    IF val_file_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SELECT object_id
      INTO val_object_id
      FROM openestate_objects_files
      WHERE file_id = val_file_id;
    SET allowed = can_write_immotool_agency_objects(val_object_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Datei ggf. aktualisieren
    IF val_file_content IS NOT NULL THEN

      -- Datei bearbeiten
      UPDATE openestate_objects_files
        SET
          file_content = val_file_content,
          file_size = val_file_size,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

    -- Vorschau ggf. aktualisieren
    IF val_file_content_preview IS NOT NULL THEN

      -- Vorschau bearbeiten
      UPDATE openestate_objects_files
        SET
          file_content_preview = val_file_content_preview,
          modified_at = NOW()
        WHERE
          file_id = val_file_id;

    END IF;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE set_openestate_objects_files_content
  TO "IMMOTOOL";

-- -----------------------------------------------------
-- Procedure set_openestate_searches_enabled
-- -----------------------------------------------------
DROP PROCEDURE IF EXISTS set_openestate_searches_enabled CASCADE;
\.
CREATE PROCEDURE set_openestate_searches_enabled(
    val_search_id BIGINT,
    val_search_enabled BOOLEAN
  )
  SPECIFIC set_openestate_searches_enabled
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_search_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Prüfen ob Schreib-Rechte vorliegen
    SET allowed = can_write_immotool_agency_searches(val_search_id);
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Suche bearbeiten
    UPDATE openestate_searches
      SET
        search_enabled = val_search_enabled,
        modified_at = NOW()
      WHERE
        search_id = val_search_id;

  END;
.;
GRANT EXECUTE
  ON PROCEDURE set_openestate_searches_enabled
  TO "IMMOTOOL";
