-- -----------------------------------------------------
-- Contacts extensions to the Agency-Addon for OpenEstate-ImmoTool
-- routines for HSQLDB
-- Copyright (C) 2009-2019 OpenEstate.org
-- -----------------------------------------------------

\.
-- -----------------------------------------------------
-- Function can_delete_immotool_agency_brokerings
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_agency_brokerings( val_brokering_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_brokerings
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen (Schreibrechte auf der zugewiesenen Immobilie)
    SELECT obj.access_permissions, obj.access_owner_id, obj.access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_brokerings brok, immotool_agency_objects obj
      WHERE brok.brokering_id = val_brokering_id AND brok.agency_object_id = obj.object_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_delete_immotool_agency_interests
-- -----------------------------------------------------
CREATE FUNCTION can_delete_immotool_agency_interests( val_interest_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_delete_immotool_agency_interests
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen (Schreibrechte auf der zugewiesenen Adresse)
    SELECT addrgrp.access_permissions, addrgrp.access_owner_id, addrgrp.access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_interests intr, immotool_contacts_entries addr, immotool_contacts_groups addrgrp
      WHERE intr.interest_id = val_interest_id AND intr.contacts_entry_id = addr.entry_id AND addr.group_id = addrgrp.group_id;

    RETURN can_write( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_agency_brokerings
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_agency_brokerings( val_brokering_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_brokerings
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen (Leserechte auf der zugewiesenen Immobilie)
    SELECT obj.access_permissions, obj.access_owner_id, obj.access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_brokerings brok, immotool_agency_objects obj
      WHERE brok.brokering_id = val_brokering_id AND brok.agency_object_id = obj.object_id;

    RETURN can_read( permissions, owner_uid, owner_gid );
  END;
.;
\.
-- -----------------------------------------------------
-- Function can_write_immotool_agency_interests
-- -----------------------------------------------------
CREATE FUNCTION can_write_immotool_agency_interests( val_interest_id BIGINT )
  RETURNS BOOLEAN
  SPECIFIC can_write_immotool_agency_interests
  READS SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    -- auf Administrator-Rechte prüfen
    IF is_admin() = TRUE THEN
      RETURN TRUE;
    END IF;

    -- auf Benutzer-Rechte prüfen (Leserechte auf der zugewiesenen Adresse)
    SELECT addrgrp.access_permissions, addrgrp.access_owner_id, addrgrp.access_group_id
      INTO permissions, owner_uid, owner_gid
      FROM immotool_agency_interests intr, immotool_contacts_entries addr, immotool_contacts_groups addrgrp
      WHERE intr.interest_id = val_interest_id AND intr.contacts_entry_id = addr.entry_id AND addr.group_id = addrgrp.group_id;

    RETURN can_read( permissions, owner_uid, owner_gid );
  END;
.;

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_agency_brokerings
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_agency_brokerings( val_brokering_id BIGINT )
  SPECIFIC remove_immotool_agency_brokerings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_brokering_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_delete_immotool_agency_brokerings( val_brokering_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag entfernen
    DELETE
      FROM immotool_agency_brokerings
      WHERE brokering_id = val_brokering_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_brokerings
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_agency_interests
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_agency_interests( val_interest_id BIGINT )
  SPECIFIC remove_immotool_agency_interests
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_interest_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_delete_immotool_agency_interests( val_interest_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Eintrag entfernen
    DELETE
      FROM immotool_agency_interests
      WHERE interest_id = val_interest_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_interests
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_agency_interests_offers
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_agency_interests_offers(
    val_interest_id BIGINT,
    val_agency_object_id BIGINT
  )
  SPECIFIC remove_immotool_agency_interests_offers
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    IF val_interest_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Schreib-Rechte besitzt
    SET allowed = can_write_immotool_agency_interests( val_interest_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF val_agency_object_id < 1 THEN

      -- alle Relationen entfernen
      DELETE
        FROM immotool_agency_interests_offers
        WHERE interest_id = val_interest_id;

    ELSE

      -- Relation entfernen
      DELETE
        FROM immotool_agency_interests_offers
        WHERE interest_id = val_interest_id
        AND agency_object_id = val_agency_object_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_interests_offers
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure remove_immotool_agency_objects_contacts_entries
-- -----------------------------------------------------
CREATE PROCEDURE remove_immotool_agency_objects_contacts_entries(
    val_agency_object_id BIGINT,
    val_contacts_entry_id BIGINT
  )
  SPECIFIC remove_immotool_agency_objects_contacts_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE permissions INT;
    DECLARE owner_uid BIGINT;
    DECLARE owner_gid BIGINT;

    IF val_agency_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    IF val_contacts_entry_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- sicherstellen, dass der Benutzer Lese-Rechte besitzt
    IF NOT is_admin() = TRUE THEN

      -- Lese-Rechte auf der Immobilie müssen vorliegen
      IF NOT can_read_immotool_agency_objects( val_agency_object_id ) = TRUE THEN
        SIGNAL SQLSTATE '45000';
      END IF;

      -- Lese-Rechte auf dem Adresseintrag müssen vorliegen
      IF NOT can_read_immotool_contacts_entries( val_contacts_entry_id ) = TRUE THEN
        SIGNAL SQLSTATE '45000';
      END IF;

    END IF;

    -- Relation entfernen
    DELETE
      FROM immotool_agency_objects_contacts_entries
      WHERE agency_object_id = val_agency_object_id
      AND contacts_entry_id = val_contacts_entry_id;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE remove_immotool_agency_objects_contacts_entries
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_agency_brokerings
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_agency_brokerings(
    INOUT val_brokering_id BIGINT,
    val_contacts_entry_id BIGINT,
    val_agency_object_id BIGINT,
    val_brokering_exclusive BOOLEAN,
    val_brokering_begin TIMESTAMP WITH TIME ZONE,
    val_brokering_end TIMESTAMP WITH TIME ZONE,
    val_brokering_notes LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_brokerings
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass Schreibrechte auf der zugewiesenen Immobilie vorliegen
    SET allowed = can_write_immotool_agency_objects( val_agency_object_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neue Vermarktung erzeugen
    IF val_brokering_id < 1 THEN

      -- Vermarktung speichern
      INSERT INTO immotool_agency_brokerings (
        contacts_entry_id,
        agency_object_id,
        brokering_exclusive,
        brokering_begin,
        brokering_end,
        brokering_notes )
      VALUES (
        val_contacts_entry_id,
        val_agency_object_id,
        val_brokering_exclusive,
        val_brokering_begin,
        val_brokering_end,
        val_brokering_notes);

      -- ID zurückliefern
      SET val_brokering_id = IDENTITY();

    -- Bestehende Vermarktung bearbeiten
    ELSE

      -- Vermarktung bearbeiten
      UPDATE immotool_agency_brokerings
        SET
          contacts_entry_id = val_contacts_entry_id,
          agency_object_id = val_agency_object_id,
          brokering_exclusive = val_brokering_exclusive,
          brokering_begin = val_brokering_begin,
          brokering_end = val_brokering_end,
          brokering_notes = val_brokering_notes,
          modified_at = NOW()
        WHERE
          brokering_id = val_brokering_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_brokerings
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_agency_interests
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_agency_interests(
    INOUT val_interest_id BIGINT,
    val_contacts_entry_id BIGINT,
    val_agency_search_id BIGINT,
    val_interest_status VARCHAR(100),
    val_interest_begin TIMESTAMP WITH TIME ZONE,
    val_interest_end TIMESTAMP WITH TIME ZONE,
    val_interest_notes LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_interests
  MODIFIES SQL DATA
  BEGIN ATOMIC
    DECLARE allowed BOOLEAN;

    -- sicherstellen, dass Schreibrechte auf der zugewiesenen Adresse vorliegen
    SET allowed = can_write_immotool_contacts_entries( val_contacts_entry_id );
    IF NOT allowed = TRUE THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Neuen Interessent erzeugen
    IF val_interest_id < 1 THEN

      -- Interessent speichern
      INSERT INTO immotool_agency_interests (
        contacts_entry_id,
        agency_search_id,
        interest_status,
        interest_begin,
        interest_end,
        interest_notes )
      VALUES (
        val_contacts_entry_id,
        val_agency_search_id,
        val_interest_status,
        val_interest_begin,
        val_interest_end,
        val_interest_notes);

      -- ID zurückliefern
      SET val_interest_id = IDENTITY();

    -- Bestehenden Interessent bearbeiten
    ELSE

      -- Interessent bearbeiten
      UPDATE immotool_agency_interests
        SET
          contacts_entry_id = val_contacts_entry_id,
          agency_search_id = val_agency_search_id,
          interest_status = val_interest_status,
          interest_begin = val_interest_begin,
          interest_end = val_interest_end,
          interest_notes = val_interest_notes,
          modified_at = NOW()
        WHERE
          interest_id = val_interest_id;

    END IF;

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_interests
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_agency_interests_offers
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_agency_interests_offers(
    val_interest_id BIGINT,
    val_agency_object_id BIGINT,
    val_offer_time TIMESTAMP WITH TIME ZONE,
    val_offer_status VARCHAR(100),
    val_offer_notes LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_interests_offers
  MODIFIES SQL DATA
  BEGIN ATOMIC

    IF val_interest_id < 1 OR val_agency_object_id < 1 THEN
      SIGNAL SQLSTATE '45000';
    END IF;

    -- Relation ggf. entfernen
    CALL remove_immotool_agency_interests_offers( val_interest_id, val_agency_object_id );

    -- Relation einfügen
    INSERT INTO immotool_agency_interests_offers (
      interest_id,
      agency_object_id,
      offer_time,
      offer_status,
      offer_notes )
    VALUES (
      val_interest_id,
      val_agency_object_id,
      val_offer_time,
      val_offer_status,
      val_offer_notes );

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_interests_offers
  TO "IMMOTOOL";

\.
-- -----------------------------------------------------
-- Procedure save_immotool_agency_objects_contacts_entries
-- -----------------------------------------------------
CREATE PROCEDURE save_immotool_agency_objects_contacts_entries(
    val_agency_object_id BIGINT,
    val_contacts_entry_id BIGINT,
    val_relation_type VARCHAR(100),
    val_relation_notes LONGVARCHAR
  )
  SPECIFIC save_immotool_agency_objects_contacts_entries
  MODIFIES SQL DATA
  BEGIN ATOMIC

    -- Relation ggf. entfernen
    CALL remove_immotool_agency_objects_contacts_entries( val_agency_object_id, val_contacts_entry_id );

    -- Relation einfügen
    INSERT INTO immotool_agency_objects_contacts_entries (
      agency_object_id,
      contacts_entry_id,
      relation_type,
      relation_notes )
    VALUES (
      val_agency_object_id,
      val_contacts_entry_id,
      val_relation_type,
      val_relation_notes );

  END;
.;

GRANT EXECUTE
  ON PROCEDURE save_immotool_agency_objects_contacts_entries
  TO "IMMOTOOL";
