-- -----------------------------------------------------
-- OpenEstate-SQL schema for HSQLDB
-- Copyright (C) 2009-2019 OpenEstate.org
-- -----------------------------------------------------

CREATE SEQUENCE seq_openestate_contacts
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_openestate_geodbs
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_openestate_geodbs_areas
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_openestate_geodbs_countries
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_openestate_languages
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_openestate_objects
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_openestate_objects_files
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_openestate_objects_links
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;
CREATE SEQUENCE seq_openestate_searches
  AS BIGINT START WITH 1 INCREMENT BY 1 NO CYCLE;


-- -----------------------------------------------------
-- Table openestate_languages
-- -----------------------------------------------------
CREATE MEMORY TABLE openestate_languages (
  language_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_languages NOT NULL,
  language_code VARCHAR(50) DEFAULT NULL NULL,
  language_enabled BOOLEAN DEFAULT true NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (language_id),
  UNIQUE (language_code)
);


-- -----------------------------------------------------
-- Table openestate_contacts
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_contacts (
  contact_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_contacts NOT NULL,
  person_first_name VARCHAR(150) DEFAULT NULL NULL,
  person_middle_name VARCHAR(150) DEFAULT NULL NULL,
  person_last_name VARCHAR(150) DEFAULT NULL NULL,
  person_title VARCHAR(50) DEFAULT NULL NULL,
  person_gender CHAR(1) DEFAULT NULL NULL,
  person_date_of_birth DATE DEFAULT NULL NULL,
  person_mail VARCHAR(150) DEFAULT NULL NULL,
  person_phone VARCHAR(50) DEFAULT NULL NULL,
  person_mobile_phone VARCHAR(50) DEFAULT NULL NULL,
  person_fax VARCHAR(50) DEFAULT NULL NULL,
  address_street VARCHAR(150) DEFAULT NULL NULL,
  address_street2 VARCHAR(150) DEFAULT NULL NULL,
  address_street_nr VARCHAR(20) DEFAULT NULL NULL,
  address_postal_code VARCHAR(10) DEFAULT NULL NULL,
  address_city VARCHAR(150) DEFAULT NULL NULL,
  address_district VARCHAR(150) DEFAULT NULL NULL,
  address_country CHAR(2) DEFAULT NULL NULL,
  address_region VARCHAR(150) DEFAULT NULL NULL,
  address_latitude DECIMAL(15,12) DEFAULT NULL NULL,
  address_longitude DECIMAL(15,12) DEFAULT NULL NULL,
  contact_website VARCHAR(255) DEFAULT NULL NULL,
  contact_office_hours LONGVARCHAR DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (contact_id)
);


-- -----------------------------------------------------
-- Table openestate_objects
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects (
  object_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_objects NOT NULL,
  contact_id BIGINT DEFAULT NULL NULL,
  object_type VARCHAR(100) NOT NULL,
  object_action VARCHAR(100) NOT NULL ,
  object_public_id VARCHAR(255) DEFAULT NULL NULL,
  object_group_id VARCHAR(255) DEFAULT NULL NULL,
  object_currency CHAR(3) NOT NULL,
  object_hidden_prices BOOLEAN DEFAULT false NOT NULL,
  object_publish_begin TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  object_publish_end TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  address_published BOOLEAN DEFAULT true NOT NULL,
  address_street VARCHAR(150) DEFAULT NULL NULL,
  address_street2 VARCHAR(150) DEFAULT NULL NULL,
  address_street_nr VARCHAR(20) DEFAULT NULL NULL,
  address_postal_code VARCHAR(10) DEFAULT NULL NULL,
  address_city VARCHAR(150) DEFAULT NULL NULL,
  address_district VARCHAR(150) DEFAULT NULL NULL,
  address_country CHAR(2) DEFAULT NULL NULL,
  address_region VARCHAR(150) DEFAULT NULL NULL,
  address_latitude DECIMAL(15,12) DEFAULT NULL NULL,
  address_longitude DECIMAL(15,12) DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (object_id),
  UNIQUE (object_public_id),
  FOREIGN KEY (contact_id)
    REFERENCES openestate_contacts(contact_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_objects_attribs
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_attribs (
  object_id BIGINT NOT NULL,
  language_id BIGINT NOT NULL,
  attrib_name VARCHAR(255) NOT NULL,
  attrib_unit VARCHAR(25) DEFAULT NULL NULL,
  value_text LONGVARCHAR DEFAULT NULL NULL,
  value_string VARCHAR(255) DEFAULT NULL NULL,
  value_boolean BOOLEAN DEFAULT NULL NULL,
  value_numeric DECIMAL(15,2) DEFAULT NULL NULL,
  value_numeric_base DECIMAL(15,2) DEFAULT NULL NULL,
  value_date DATE DEFAULT NULL NULL,
  value_time TIME DEFAULT NULL NULL,
  value_datetime TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (object_id, language_id, attrib_name),
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (language_id)
    REFERENCES openestate_languages(language_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_objects_i18n
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_i18n (
  object_id BIGINT NOT NULL,
  language_id BIGINT NOT NULL,
  i18n_title VARCHAR(255) DEFAULT NULL NULL,
  PRIMARY KEY (object_id, language_id),
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (language_id)
    REFERENCES openestate_languages(language_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_objects_files
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_files (
  file_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_objects_files NOT NULL,
  object_id BIGINT NOT NULL,
  file_name VARCHAR(255) DEFAULT NULL NULL,
  file_url VARCHAR(255) DEFAULT NULL NULL,
  file_content BLOB(1G) DEFAULT NULL NULL,
  file_content_preview BLOB(1M) DEFAULT NULL NULL,
  file_mimetype VARCHAR(255) NOT NULL,
  file_published BOOLEAN DEFAULT true NOT NULL,
  file_type VARCHAR(100) DEFAULT NULL NULL,
  file_size BIGINT DEFAULT 0 NOT NULL,
  file_hash VARCHAR(40) DEFAULT NULL NULL,
  file_image_width INT DEFAULT 0 NOT NULL,
  file_image_height INT DEFAULT 0 NOT NULL,
  file_position INT DEFAULT 0 NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (file_id),
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_objects_files_i18n
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_files_i18n (
  file_id BIGINT NOT NULL,
  language_id BIGINT NOT NULL,
  i18n_title VARCHAR(255) DEFAULT NULL NULL,
  PRIMARY KEY (file_id, language_id),
  FOREIGN KEY (file_id)
    REFERENCES openestate_objects_files(file_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (language_id)
    REFERENCES openestate_languages(language_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_objects_links
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_links (
  link_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_objects_links NOT NULL,
  object_id BIGINT NOT NULL ,
  link_web_url VARCHAR(255) DEFAULT NULL NULL,
  link_provider_id VARCHAR(255) DEFAULT NULL NULL,
  link_provider_name VARCHAR(255) DEFAULT NULL NULL,
  link_published BOOLEAN DEFAULT TRUE NOT NULL,
  link_position INT DEFAULT 0 NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (link_id),
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_objects_links_i18n
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_links_i18n (
  link_id BIGINT NOT NULL ,
  language_id BIGINT NOT NULL ,
  i18n_title VARCHAR(255) DEFAULT NULL NULL,
  PRIMARY KEY (link_id, language_id),
  FOREIGN KEY (link_id)
    REFERENCES openestate_objects_links(link_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (language_id)
    REFERENCES openestate_languages(language_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_searches
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_searches (
  search_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_searches NOT NULL,
  search_name VARCHAR(255) DEFAULT NULL NULL,
  search_notes LONGVARCHAR DEFAULT NULL NULL,
  search_enabled BOOLEAN DEFAULT TRUE NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (search_id)
);


-- -----------------------------------------------------
-- Table openestate_searches_values
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_searches_values (
  search_id BIGINT NOT NULL,
  value_type VARCHAR(100) NOT NULL,
  value_equals VARCHAR(255) DEFAULT NULL NULL,
  value_equals_like BOOLEAN DEFAULT false NOT NULL,
  value_equals_case BOOLEAN DEFAULT false NOT NULL,
  value_order BIGINT DEFAULT 0 NOT NULL,
  FOREIGN KEY (search_id)
    REFERENCES openestate_searches(search_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_searches_attribs
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_searches_attribs (
  search_id BIGINT NOT NULL,
  attrib_name VARCHAR(255) NOT NULL,
  attrib_equals VARCHAR(255) DEFAULT NULL NULL,
  attrib_equals_like BOOLEAN DEFAULT NULL NULL,
  attrib_equals_case BOOLEAN DEFAULT NULL NULL,
  attrib_matches VARCHAR(255) DEFAULT NULL NULL,
  attrib_numeric_min DECIMAL(15,2) DEFAULT NULL NULL,
  attrib_numeric_max DECIMAL(15,2) DEFAULT NULL NULL,
  attrib_numeric_unit VARCHAR(20) DEFAULT NULL NULL,
  attrib_time_begin TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  attrib_time_end TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  attrib_order BIGINT DEFAULT 0 NOT NULL,
  FOREIGN KEY (search_id)
    REFERENCES openestate_searches(search_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_searches_params
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_searches_params (
  search_id BIGINT NOT NULL,
  param_key VARCHAR(255) NOT NULL,
  param_value VARCHAR(255) DEFAULT NULL NULL,
  PRIMARY KEY (search_id, param_key),
  FOREIGN KEY (search_id)
    REFERENCES openestate_searches(search_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_geodbs
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_geodbs (
  geodb_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_geodbs NOT NULL,
  geodb_name VARCHAR(255) NOT NULL,
  geodb_provider VARCHAR(255) NOT NULL,
  geodb_build_time TIMESTAMP WITH TIME ZONE DEFAULT NULL NULL,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  modified_at TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
  PRIMARY KEY (geodb_id),
  UNIQUE (geodb_name)
);


-- -----------------------------------------------------
-- Table openestate_geodbs_countries
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_geodbs_countries (
  country_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_geodbs_countries NOT NULL,
  geodb_id BIGINT NOT NULL,
  country_iso CHAR(2) NOT NULL,
  country_name VARCHAR(255) DEFAULT NULL NULL,
  country_provider_id VARCHAR(255) DEFAULT NULL NULL,
  PRIMARY KEY (country_id),
  UNIQUE (geodb_id, country_iso),
  FOREIGN KEY (geodb_id)
    REFERENCES openestate_geodbs(geodb_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_geodbs_areas
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_geodbs_areas (
  area_id BIGINT GENERATED BY DEFAULT AS SEQUENCE seq_openestate_geodbs_areas NOT NULL,
  country_id BIGINT NOT NULL,
  area_parent_id BIGINT DEFAULT NULL NULL,
  area_name VARCHAR(255) NOT NULL,
  area_level VARCHAR(15) DEFAULT NULL NULL,
  area_provider_id VARCHAR(255) DEFAULT NULL NULL,
  PRIMARY KEY (area_id),
  FOREIGN KEY (country_id)
    REFERENCES openestate_geodbs_countries(country_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (area_parent_id)
    REFERENCES openestate_geodbs_areas(area_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);


-- -----------------------------------------------------
-- Table openestate_objects_geodbs_areas
-- -----------------------------------------------------
CREATE CACHED TABLE openestate_objects_geodbs_areas (
  object_id BIGINT NOT NULL,
  area_id BIGINT NOT NULL,
  PRIMARY KEY (object_id, area_id),
  FOREIGN KEY (object_id)
    REFERENCES openestate_objects(object_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE,
  FOREIGN KEY (area_id)
    REFERENCES openestate_geodbs_areas(area_id)
    MATCH FULL
    ON UPDATE CASCADE
    ON DELETE CASCADE
);

-- -----------------------------------------------------
-- View view_openestate_contacts
-- -----------------------------------------------------
CREATE VIEW view_openestate_contacts AS
  SELECT * FROM openestate_contacts;


-- -----------------------------------------------------
-- View view_openestate_languages
-- -----------------------------------------------------
CREATE VIEW view_openestate_languages AS
  SELECT * FROM openestate_languages;


-- -----------------------------------------------------
-- View view_openestate_objects
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects AS
  SELECT * FROM openestate_objects;


-- -----------------------------------------------------
-- View view_openestate_objects_actions
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_actions AS
  SELECT object_action FROM openestate_objects GROUP BY object_action;


-- -----------------------------------------------------
-- View view_openestate_objects_attribs
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_attribs AS
  SELECT * FROM openestate_objects_attribs;


-- -----------------------------------------------------
-- View view_openestate_objects_files
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_files AS
  SELECT
    file_id,
    object_id,
    file_name,
    file_url,
    file_mimetype,
    file_published,
    file_type,
    file_size,
    file_hash,
    file_image_width,
    file_image_height,
    file_position,
    created_at,
    modified_at
  FROM openestate_objects_files;


-- -----------------------------------------------------
-- View view_openestate_objects_files_content
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_files_content AS
  SELECT
    file_id,
    file_content,
    created_at,
    modified_at
  FROM openestate_objects_files;


-- -----------------------------------------------------
-- View view_openestate_objects_files_i18n
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_files_i18n AS
  SELECT * FROM openestate_objects_files_i18n;


-- -----------------------------------------------------
-- View view_openestate_objects_files_preview
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_files_preview AS
  SELECT
    file_id,
    file_content_preview,
    created_at,
    modified_at
  FROM openestate_objects_files;


-- -----------------------------------------------------
-- View view_openestate_objects_links
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_links AS
  SELECT * FROM openestate_objects_links;


-- -----------------------------------------------------
-- View view_openestate_objects_links_i18n
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_links_i18n AS
  SELECT * FROM openestate_objects_links_i18n;


-- -----------------------------------------------------
-- View view_openestate_objects_geodbs_areas
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_geodbs_areas AS
  SELECT a.*, c.geodb_id
  FROM openestate_objects_geodbs_areas a
  INNER JOIN openestate_geodbs_areas b ON (a.area_id = b.area_id)
  INNER JOIN openestate_geodbs_countries c ON (b.country_id = c.country_id);


-- -----------------------------------------------------
-- View view_openestate_objects_i18n
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_i18n AS
  SELECT * FROM openestate_objects_i18n;


-- -----------------------------------------------------
-- View view_openestate_objects_types
-- -----------------------------------------------------
CREATE VIEW view_openestate_objects_types AS
  SELECT object_type FROM openestate_objects GROUP BY object_type;


-- -----------------------------------------------------
-- View view_openestate_searches
-- -----------------------------------------------------
CREATE VIEW view_openestate_searches AS
  SELECT * FROM openestate_searches;


-- -----------------------------------------------------
-- View view_openestate_searches_attribs
-- -----------------------------------------------------
CREATE VIEW view_openestate_searches_attribs AS
  SELECT * FROM openestate_searches_attribs;


-- -----------------------------------------------------
-- View view_openestate_searches_params
-- -----------------------------------------------------
CREATE VIEW view_openestate_searches_params AS
  SELECT * FROM openestate_searches_params;


-- -----------------------------------------------------
-- View view_openestate_searches_values
-- -----------------------------------------------------
CREATE VIEW view_openestate_searches_values AS
  SELECT * FROM openestate_searches_values;


-- -----------------------------------------------------
-- View view_openestate_geodbs
-- -----------------------------------------------------
CREATE VIEW view_openestate_geodbs AS
  SELECT * FROM openestate_geodbs;


-- -----------------------------------------------------
-- View view_openestate_geodbs_countries
-- -----------------------------------------------------
CREATE VIEW view_openestate_geodbs_countries AS
  SELECT * FROM openestate_geodbs_countries;


-- -----------------------------------------------------
-- View view_openestate_geodbs_areas
-- -----------------------------------------------------
CREATE VIEW view_openestate_geodbs_areas AS
  SELECT * FROM openestate_geodbs_areas;
